/*
 * Decompiled with CFR 0.152.
 */
package org.asamk.signal.manager.storage.messageCache;

import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.Arrays;
import java.util.Collections;
import java.util.Objects;
import java.util.stream.Stream;
import org.asamk.signal.manager.storage.messageCache.CachedMessage;
import org.asamk.signal.manager.storage.recipients.RecipientId;
import org.asamk.signal.manager.util.IOUtils;
import org.asamk.signal.manager.util.MessageCacheUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.whispersystems.signalservice.api.messages.SignalServiceEnvelope;

public class MessageCache {
    private static final Logger logger = LoggerFactory.getLogger(MessageCache.class);
    private final File messageCachePath;

    public MessageCache(File messageCachePath) {
        this.messageCachePath = messageCachePath;
    }

    public Iterable<CachedMessage> getCachedMessages() {
        if (!this.messageCachePath.exists()) {
            return Collections.emptyList();
        }
        return Arrays.stream(Objects.requireNonNull(this.messageCachePath.listFiles())).flatMap(dir -> {
            if (dir.isFile()) {
                return Stream.of(dir);
            }
            File[] files = Objects.requireNonNull(dir.listFiles());
            if (files.length == 0) {
                try {
                    Files.delete(dir.toPath());
                }
                catch (IOException e) {
                    logger.warn("Failed to delete cache dir \u201c{}\u201d, ignoring: {}", dir, (Object)e.getMessage());
                }
                return Stream.empty();
            }
            return Arrays.stream(files).filter(File::isFile);
        }).map(CachedMessage::new).toList();
    }

    public CachedMessage cacheMessage(SignalServiceEnvelope envelope, RecipientId recipientId) {
        File cacheFile;
        long now = System.currentTimeMillis();
        try {
            cacheFile = this.getMessageCacheFile(recipientId, now, envelope.getTimestamp());
        }
        catch (IOException e) {
            logger.warn("Failed to create recipient folder in disk cache: {}", (Object)e.getMessage());
            throw new RuntimeException(e);
        }
        CachedMessage cachedMessage = new CachedMessage(cacheFile, envelope);
        try {
            MessageCacheUtils.storeEnvelope(envelope, cacheFile);
            return cachedMessage;
        }
        catch (IOException e) {
            logger.warn("Failed to store encrypted message in disk cache, ignoring: {}", (Object)e.getMessage());
            return cachedMessage;
        }
    }

    public CachedMessage replaceSender(CachedMessage cachedMessage, RecipientId sender) throws IOException {
        File cacheFile = this.getMessageCacheFile(sender, cachedMessage.getFile().getName());
        if (cacheFile.equals(cachedMessage.getFile())) {
            return cachedMessage;
        }
        logger.debug("Moving cached message {} to {}", (Object)cachedMessage.getFile().toPath(), (Object)cacheFile.toPath());
        Files.move(cachedMessage.getFile().toPath(), cacheFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        return new CachedMessage(cacheFile);
    }

    public void deleteMessages(RecipientId recipientId) {
        File recipientMessageCachePath = this.getMessageCachePath(recipientId);
        if (!recipientMessageCachePath.exists()) {
            return;
        }
        for (File file : Objects.requireNonNull(recipientMessageCachePath.listFiles())) {
            if (!file.isFile()) continue;
            try {
                Files.delete(file.toPath());
            }
            catch (IOException e) {
                logger.warn("Failed to delete cache file \u201c{}\u201d, ignoring: {}", (Object)file, (Object)e.getMessage());
            }
        }
    }

    private File getMessageCachePath(RecipientId recipientId) {
        if (recipientId == null) {
            return this.messageCachePath;
        }
        String sender = String.valueOf(recipientId.id());
        return new File(this.messageCachePath, sender.replace("/", "_"));
    }

    private File getMessageCacheFile(RecipientId recipientId, String filename) throws IOException {
        File cachePath = this.getMessageCachePath(recipientId);
        IOUtils.createPrivateDirectories(cachePath);
        return new File(cachePath, filename);
    }

    private File getMessageCacheFile(RecipientId recipientId, long now, long timestamp) throws IOException {
        File cachePath = this.getMessageCachePath(recipientId);
        IOUtils.createPrivateDirectories(cachePath);
        return new File(cachePath, now + "_" + timestamp);
    }

    public void mergeRecipients(RecipientId recipientId, RecipientId toBeMergedRecipientId) {
        File toBeMergedMessageCachePath = this.getMessageCachePath(toBeMergedRecipientId);
        if (!toBeMergedMessageCachePath.exists()) {
            return;
        }
        for (File file : Objects.requireNonNull(toBeMergedMessageCachePath.listFiles())) {
            if (!file.isFile()) continue;
            try {
                File cacheFile = this.getMessageCacheFile(recipientId, file.getName());
                Files.move(file.toPath(), cacheFile.toPath(), new CopyOption[0]);
            }
            catch (IOException e) {
                logger.warn("Failed to move cache file \u201c{}\u201d, ignoring: {}", new Object[]{file, e.getMessage(), e});
            }
        }
    }
}

