use nettle_sys::{
    nettle_hash, nettle_sha3_224, nettle_sha3_224_digest, nettle_sha3_224_init,
    nettle_sha3_224_update, sha3_224_ctx,
};
use std::default::Default;
use std::mem::zeroed;

use crate::hash::NettleHash;
use crate::hash::Hash;

/// 224 bit variant of the Secure Hash Algorithm 3 (SHA-3) defined in FIPS 202.
#[allow(non_camel_case_types)]
pub struct Sha3_224 {
    context: sha3_224_ctx,
}

impl_write_for_hash!(Sha3_224);

impl Clone for Sha3_224 {
    fn clone(&self) -> Self {
        use std::intrinsics::copy_nonoverlapping;

        unsafe {
            let mut ctx: sha3_224_ctx = zeroed();
            copy_nonoverlapping(&self.context, &mut ctx, 1);

            Sha3_224 { context: ctx }
        }
    }
}

impl Default for Sha3_224 {
    fn default() -> Self {
        let mut ctx = unsafe { zeroed() };

        unsafe {
            nettle_sha3_224_init(&mut ctx as *mut _);
        }

        Sha3_224 { context: ctx }
    }
}

impl Hash for Sha3_224 {
    fn digest_size(&self) -> usize {
        ::nettle_sys::SHA3_224_DIGEST_SIZE as usize
    }

    fn update(&mut self, data: &[u8]) {
        unsafe {
            nettle_sha3_224_update(
                &mut self.context as *mut _,
                data.len(),
                data.as_ptr(),
            );
        }
    }

    fn digest(&mut self, digest: &mut [u8]) {
        unsafe {
            nettle_sha3_224_digest(
                &mut self.context as *mut _,
                digest.len(),
                digest.as_mut_ptr(),
            );
        }
    }

    fn box_clone(&self) -> Box<dyn Hash> {
        Box::new(self.clone())
    }
}

impl NettleHash for Sha3_224 {
    type Context = sha3_224_ctx;

    unsafe fn nettle_hash() -> &'static nettle_hash {
        &nettle_sha3_224
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn do_nothing() {
        let _ = Sha3_224::default();
    }

    //  CAVS 19.0
    //  "SHA3-224 ShortMsg" information for "SHA3AllBytes1-28-16"
    //  SHA3-224 tests are configured for BYTE oriented implementations
    //  Length values represented in bits
    //  Generated on Thu Jan 28 13:32:43 2016
    #[test]
    fn nist_cavs_short_msg() {
        let mut ctx = Sha3_224::default();
        let mut digest = vec![0u8; ctx.digest_size()];

        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6b\x4e\x03\x42\x36\x67\xdb\xb7\x3b\x6e\x15\x45\x4f\x0e\xb1\xab\xd4\x59\x7f\x9a\x1b\x07\x8e\x3f\x5b\x5a\x6b\xc7"[..]);

        ctx.update(b"\x01");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x48\x82\x86\xd9\xd3\x27\x16\xe5\x88\x1e\xa1\xee\x51\xf3\x6d\x36\x60\xd7\x0f\x0d\xb0\x3b\x3f\x61\x2c\xe9\xed\xa4"[..]);

        ctx.update(b"\x69\xcb");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x94\xbd\x25\xc4\xcf\x6c\xa8\x89\x12\x6d\xf3\x7d\xdd\x9c\x36\xe6\xa9\xb2\x8a\x4f\xe1\x5c\xc3\xda\x6d\xeb\xcd\xd7"[..]);

        ctx.update(b"\xbf\x58\x31");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x1b\xb3\x6b\xeb\xde\x5f\x3c\xb6\xd8\xe4\x67\x2a\xcf\x6e\xec\x87\x28\xf3\x1a\x54\xda\xcc\x25\x60\xda\x2a\x00\xcc"[..]);

        ctx.update(b"\xd1\x48\xce\x6d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x0b\x52\x1d\xac\x1e\xfe\x29\x2e\x20\xdf\xb5\x85\xc8\xbf\xf4\x81\x89\x9d\xf7\x2d\x59\x98\x33\x15\x95\x83\x91\xba"[..]);

        ctx.update(b"\x91\xc7\x10\x68\xf8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x98\x9f\x01\x77\x09\xf5\x0b\xd0\x23\x06\x23\xc4\x17\xf3\xda\xf1\x94\x50\x7f\x7b\x90\xa1\x11\x27\xba\x16\x38\xfa"[..]);

        ctx.update(b"\xe7\x18\x3e\x4d\x89\xc9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x65\x06\x18\xf3\xb9\x45\xc0\x7d\xe8\x5b\x84\x78\xd6\x96\x09\x64\x7d\x5e\x2a\x43\x2c\x6b\x15\xfb\xb3\xdb\x91\xe4"[..]);

        ctx.update(b"\xd8\x5e\x47\x0a\x7c\x69\x88");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8a\x13\x4c\x33\xc7\xab\xd6\x73\xcd\x3d\x0c\x33\x95\x67\x00\x76\x0d\xe9\x80\xc5\xae\xe7\x4c\x96\xe6\xba\x08\xb2"[..]);

        ctx.update(b"\xe4\xea\x2c\x16\x36\x6b\x80\xd6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x7d\xd1\xa8\xe3\xff\xe8\xc9\x9c\xc5\x47\xa6\x9a\xf1\x4b\xd6\x3b\x15\xac\x26\xbd\x3d\x36\xb8\xa9\x95\x13\xe8\x9e"[..]);

        ctx.update(b"\xb2\x93\x73\xf6\xf8\x83\x9b\xd4\x98");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe0\x2a\x13\xfa\x47\x70\xf8\x24\xbc\xd6\x97\x99\x28\x48\x78\xf1\x9b\xfd\xc8\x33\xac\x6d\x86\x5f\x28\xb7\x57\xd0"[..]);

        ctx.update(b"\x49\xec\x72\xc2\x9b\x63\x03\x6d\xbe\xcd");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x47\xca\xb4\x46\x18\xf6\x2d\xd4\x31\xcc\xb1\x3b\x3b\x9c\xd9\x85\xd8\x16\xc5\xd6\x02\x6a\xfc\x38\xa2\x81\xaa\x00"[..]);

        ctx.update(b"\x50\x2f\x4e\x28\xa6\xfe\xb4\xc6\xa1\xcc\x47");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xbb\xe6\x1d\x85\xb4\xca\xe7\x16\x32\x9e\x2b\xcc\x40\x38\xe2\x82\xb4\xd7\x83\x6e\xb8\x46\x22\x88\x35\xf6\x53\x08"[..]);

        ctx.update(b"\xe7\x23\xc6\x4b\x22\x58\xb5\x12\x4f\x88\x40\x5f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd0\x9d\xa0\x94\xcf\xef\xaa\xd4\x6b\x7b\x33\x58\x30\xa9\x30\x55\x70\xf4\xf4\xaf\xe7\x9f\x86\x29\xff\x9d\x0c\x3d"[..]);

        ctx.update(b"\x0d\x51\x2e\xce\xb7\x4d\x8a\x04\x75\x31\xc1\xf7\x16");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x29\xae\x07\x44\x05\x1e\x55\x16\x71\x76\x31\x7e\xb1\x78\x50\xa2\x29\x39\xd8\xd9\x4e\xbb\x0a\x90\xb6\xd9\x8f\xde"[..]);

        ctx.update(b"\x3b\x9a\xb7\x6a\x23\xae\x56\x34\x0b\x5f\x4b\x80\xe1\xf3");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc0\x90\x3b\xe9\x6f\x38\x05\x1c\xfc\x2a\x5a\xd2\x56\xaa\x0b\x83\x32\x21\x7f\x45\x0e\xab\x90\x4e\xe8\x4b\x65\x41"[..]);

        ctx.update(
            b"\xe9\xfe\xf7\x51\xa2\x02\x97\xad\x19\x38\x66\x2d\x13\x1e\x7a",
        );
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x48\xeb\xa3\x6d\xfe\x05\x75\x59\x7d\x13\xca\x26\x13\x32\x67\x19\x9d\xae\x76\xd6\x3d\x1b\x9e\x96\x12\x72\x0d\x08"[..]);

        ctx.update(
            b"\x2b\xbb\x42\xb9\x20\xb7\xfe\xb4\xe3\x96\x2a\x15\x52\xcc\x39\x0f",
        );
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x0d\xfa\x61\xf6\xb4\x39\xbf\x8e\x3a\x6f\x37\x8f\xe3\x0a\x41\x34\xe8\xb2\xdf\xb6\x52\x99\x7a\x2a\x76\xc2\x78\x9f"[..]);

        ctx.update(b"\x22\x54\xe1\x00\xbd\xe9\x29\x50\x93\x56\x5a\x94\x87\x7c\x21\xd0\x5a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x69\x65\x25\x64\x63\x27\x6d\xbb\x26\xad\x34\xa3\x78\xc4\xba\xca\xea\xe7\x9d\x70\x02\x83\xb1\x88\xd4\x4d\x73\xeb"[..]);

        ctx.update(b"\x78\x4e\xf7\xad\xec\xbb\x9a\x4c\xb5\xac\x1d\xf8\x51\x3d\x87\xae\x97\x72");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe9\x18\xa5\xd5\x2a\x0d\x42\xab\x8b\xa2\xea\x38\x6e\xb6\xad\x83\xcb\x8d\xd9\xa6\xbd\x46\x15\x06\xbe\x35\x6e\xad"[..]);

        ctx.update(b"\xf4\xe6\x89\x64\xf7\x84\xfe\x5c\x4d\x0e\x00\xbb\x46\x22\x04\x2f\xa7\x04\x8e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x76\x5f\x05\x0c\x95\xae\x33\x47\xcf\x3f\x4f\x50\x32\xb4\x28\xfa\xea\xb1\x36\x94\xe8\xc7\x79\x8e\xaf\xb8\x24\x75"[..]);

        ctx.update(b"\xa9\xca\x7e\xc7\xaa\xf8\x9d\xb3\x52\xfe\xcb\xa6\x46\xff\x73\xef\xe8\xe4\xa7\xe8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x65\xd6\xa4\x97\x39\xc0\xe2\x87\x58\x4f\xf9\xd1\xf3\x46\x3c\xe2\xe5\x55\xae\x96\x78\x14\x7e\x21\xb5\x88\x9e\x98"[..]);

        ctx.update(b"\xb2\xf7\x01\x85\x81\xa4\xe4\x59\xcf\x9b\x9d\x98\x16\xfc\x17\x90\x3b\xa8\x03\x3f\x13");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc6\x83\x7f\x12\x22\x7b\xfb\xd8\x6c\xcf\xe7\x94\x05\x3c\xe3\xa5\x40\x52\xc8\xca\x84\x30\xf5\x26\xfd\x64\xb5\xf2"[..]);

        ctx.update(b"\xf5\x00\x86\xb4\xdc\x7b\xca\x0b\xae\xc0\x07\x6a\x87\x8d\xd8\x95\x71\xd5\x2e\x47\x85\x5b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe3\x9a\xa9\x6f\xad\x58\x19\x61\xbd\xa0\x32\xed\x33\xdc\xe3\x6d\xef\xde\x95\x8b\xaf\x9b\xae\x5d\xc5\x58\xcf\x89"[..]);

        ctx.update(b"\x6e\x6e\xf9\x63\xf5\x00\x0d\x0b\x91\xb0\xad\x53\x7d\xdc\x96\x97\xf8\xdb\x8f\x10\xa3\xd5\xee");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x66\xdc\xb2\x92\xb4\xd6\xbb\x4c\xdd\x40\x99\xb8\xe7\xbf\xea\x96\x58\x68\x0c\x92\xc5\x15\x62\xc0\x91\x57\x70\x56"[..]);

        ctx.update(b"\x12\xa7\xb1\xa7\x3b\x0b\x26\xa6\x63\x62\xec\x2a\x91\xea\x5f\xf1\x1a\xf4\x9a\x7a\x14\x8a\x8c\xc5");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6f\xc9\x1e\xc8\xad\x44\x81\x73\xf5\x91\xb8\x65\xed\x3e\xb8\x91\x15\xa2\x78\x00\x33\x76\x52\x3c\x00\xe2\x2f\x2a"[..]);

        ctx.update(b"\x8a\x47\x68\xad\xd4\xa9\xbd\x7b\x3f\x27\x46\x12\x20\xce\xae\x02\x18\xcf\x33\x22\xf4\xd2\xa9\x80\xd1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9a\x88\xbc\x64\xe7\x43\xf2\xac\xaa\x16\x70\xcc\xa7\xe2\x01\xa2\x99\xe1\xcc\xe6\xdf\x70\x15\xb0\xd2\x53\x52\x13"[..]);

        ctx.update(b"\x5c\x5b\x8c\x19\x02\xc8\x60\x8c\x20\x4e\x72\xa8\x13\xe2\xb6\x25\x02\x1b\x31\x82\xc4\x8b\x00\xf7\xfe\x4f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x31\x80\x2a\x0f\xa9\xae\x7a\xe8\x86\x26\x60\x4a\xd9\xae\x41\x38\x1d\x6f\x7c\x3c\x90\xef\xfc\xfc\xf7\x0e\xfc\xf7"[..]);

        ctx.update(b"\xe8\x9e\x5c\xf0\x7a\xfb\x4a\x58\xeb\xee\xe1\x7f\xf5\x96\xd9\x0b\x32\x74\xba\x34\x8f\x14\xf2\x84\xff\xf0\x25");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x3b\xc9\xb7\x97\x3f\x55\x73\x5b\x61\x2d\xde\xe8\xcc\x79\x07\xa3\xf1\x42\x9b\x06\xdf\x7c\xb1\x29\x3b\x98\x98\x02"[..]);

        ctx.update(b"\xeb\x9e\x11\x43\x78\x2a\x0f\x9f\xa8\x15\x26\x1c\x2a\xdc\x27\x58\xfb\x1d\x88\xff\xe4\x0a\x0a\xe1\x44\x18\x9a\x48");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9d\x70\xd2\x25\x20\x09\x4a\x11\x32\x97\xa1\x92\xea\xd3\x3e\x31\x69\x24\xfd\xc7\xa2\xa9\xf8\xea\x70\x98\xb8\x4b"[..]);

        ctx.update(b"\xc4\xba\x3b\xff\x88\x5f\xb7\x83\x57\x22\x1a\x9a\x90\x3b\xc7\xeb\xd1\x1c\x77\x1f\xaf\x57\x89\xe5\xaa\xbc\x99\x3a\x5f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x7b\x02\x12\xb4\xee\x0b\x14\xdb\xa6\x2c\x2d\xb7\xa7\x65\xac\x56\xdb\x46\xe0\xb0\x6e\xb7\x44\xee\x35\x72\x6d\xdd"[..]);

        ctx.update(b"\x07\x81\x0e\x6b\x78\x51\x77\xe5\x2d\x0f\xea\xc0\x39\x4f\x3e\xcc\x41\xf3\x5a\xa0\x8f\xf1\xed\x81\x62\x57\x5f\x85\x88\x8c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb4\x13\xd6\xf0\xcc\xe1\x4b\x7a\x10\x44\xa1\x4b\xb2\x80\x3d\x53\xbe\xf9\x07\x09\x37\x69\xa5\xaa\x63\xa8\xe3\x16"[..]);

        ctx.update(b"\x01\xc7\x42\xdc\x9a\xb0\xb0\x5d\xf9\x25\xd4\xa3\x51\xe3\x8b\xea\x7c\xa7\xad\x78\x35\x94\xe2\x24\x87\xd5\xb8\x19\x85\x83\xf3");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc4\x2c\x70\x7d\xdc\x7b\x63\x09\x39\x54\x4a\xdb\xdb\xe5\x67\xa3\x33\xac\x88\xc3\xb5\xe7\x38\xde\xe8\xf8\x62\xbe"[..]);

        ctx.update(b"\xdd\x0f\x85\xb5\x5f\xdf\x56\xba\x25\x4e\x06\xf8\xc2\xb6\x50\xcc\x6b\x86\xbf\x28\xa1\x4d\x71\x40\x11\x14\x1a\x86\xb8\xf1\x4b\xd9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x0f\xe9\x24\x69\x29\x7c\x2c\x34\x91\x1e\xae\x42\x47\x10\xdb\x6d\x31\x20\x47\x89\x8b\x97\x56\xed\xc5\xc2\xde\xb2"[..]);

        ctx.update(b"\xdd\xf4\x8f\x4c\xdc\x85\x6c\x44\x83\x26\x09\x2d\xcf\x6b\xfc\x4e\xbc\xf4\xb3\x6f\xc2\xe5\x16\xeb\xa0\x95\x68\x07\x58\x8b\x6e\x82\x7b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6c\xd8\x3b\xa7\x0e\x1b\xd3\x87\xd6\x03\xab\x14\xc9\xfd\xcb\xf9\x86\x2d\x2e\xbf\x09\x87\x21\x5f\x01\x1a\xbe\xe8"[..]);

        ctx.update(b"\xc3\x4d\x1f\x87\x29\x66\x35\x69\x56\x9f\x87\xb1\xfd\x6e\x0b\x95\x4a\xe2\xe3\xb7\x23\xd6\xc9\xfc\xae\x6a\xb0\x9b\x13\xb4\xa8\x74\x83\xb2");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe5\x7e\x1d\x24\xdb\xd9\xa3\x0a\xb3\x11\x29\x1f\x5d\x6a\x95\x53\x0c\xaa\x02\x9c\x42\x1d\xde\x0b\x48\x7a\x57\x7e"[..]);

        ctx.update(b"\x80\x8d\xe7\xcb\xf8\xd8\x31\xad\x4f\x17\xeb\x58\x03\x1d\xae\xd3\x8b\xda\xb8\x2f\x46\x7f\x87\xc6\xb2\xe3\xa7\xc5\xde\x25\xc8\xe8\x22\x94\x13");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb3\xc1\x3f\x11\x22\x7f\x43\x86\xaf\xdc\xf7\x66\x3a\x12\x09\x90\xf2\x7d\xa2\x05\xff\xb9\xbf\x83\x67\x6f\x86\xdc"[..]);

        ctx.update(b"\x52\x04\xa0\xa6\x37\x07\xbd\x1c\xab\x67\xa8\x79\x79\x94\xa0\x52\xee\x73\x88\x4b\x32\x5f\xdf\x37\xd8\x6e\xf2\x80\xb3\xf5\x50\xc9\xeb\x4e\x7f\xd3");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6a\xa1\x06\x0f\x84\x12\x7b\xf2\xc9\x88\x23\x0a\x90\x72\x42\xe7\xd6\x97\x2a\x01\xc6\x77\x2b\xa0\xf7\xb8\xbc\x86"[..]);

        ctx.update(b"\xda\x94\x39\xbd\x09\x0d\xfc\x2e\xcc\xc1\x20\x3a\x7a\x82\xc5\xd6\x46\x7f\xec\x4e\x5b\x0a\x2b\x2c\x2b\x9e\xa6\x5b\x03\x20\x3a\x8c\xe3\x65\xfb\xd9\x8e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe8\xf0\x92\x9f\x1f\x62\x09\xd4\x11\x85\x29\x2d\x35\xeb\xbf\x5a\x3b\xfe\x54\x92\x71\x3b\x06\xd5\x65\x79\x45\x8d"[..]);

        ctx.update(b"\x66\x8b\xbd\x38\xc0\xad\x08\x81\xa7\xf0\x95\x15\x7d\x00\xf2\x9b\x57\x6b\x01\xba\x54\xa8\xf1\x39\x2e\x58\x6c\x64\x0e\xcb\x12\xb2\xa5\xc6\x27\xa6\x78\x84");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x75\xdd\x05\x69\x62\xc5\xbb\x5d\x6f\x61\x6a\x9f\x57\x89\x29\x92\x94\x6d\x04\x8d\xf5\x7c\x0a\x36\xa4\x0a\x36\x5a"[..]);

        ctx.update(b"\xd6\x3a\xc3\xbc\xfe\xe3\xa5\xbc\x50\x3c\xf2\x0f\xe8\xff\x49\x6b\xf7\xa8\x06\x47\x69\x87\x0c\x8f\xc5\x14\xc2\x9b\x55\x82\x5b\x62\x88\x97\x5b\xeb\x94\xba\x56");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc6\x94\xda\x94\x1a\x7a\x50\x6c\xef\x47\x1f\xdf\xfb\x52\x30\xbb\x6c\x3c\xd2\x71\x53\x41\x03\x3a\xb7\x26\x8e\x9b"[..]);

        ctx.update(b"\x98\x5f\x06\x12\x1a\xed\x60\x31\x71\x02\x0b\xad\xc2\x07\x5f\xd3\x32\x56\xd6\x7d\x40\x43\x08\x39\x57\x5d\xda\xa7\xa3\xf1\xf2\x23\x25\xd0\x6e\xa4\x02\x52\xd5\xe4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x29\xf8\x84\x6a\xaf\x23\x42\x81\xb5\x15\xea\x1d\x45\x67\x45\x35\xa6\x12\x6c\x38\xbd\x95\x9c\x19\x95\xca\xd7\xc9"[..]);

        ctx.update(b"\x87\x83\x84\x95\x52\xbe\x45\x40\xcb\x24\xd6\x79\x96\xa1\x0d\x16\x44\x4b\x2d\x93\x6d\x2f\xa5\xfc\xff\x51\xfb\x0d\xd5\xee\x03\x99\x8c\x04\x54\x28\x92\x15\xfc\xe4\x7f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x84\x50\x22\x56\xe3\xf4\x29\x1e\xf4\xd1\x5e\x87\x05\xe5\x79\x95\x1f\xc0\xe3\x9a\x2d\x58\xfd\xa7\x48\x52\x55\x1f"[..]);

        ctx.update(b"\xda\xb3\x1c\x7b\x3f\x40\x82\x5a\xac\x13\xf6\x77\x27\x71\xb7\xe7\xfb\xc0\x9f\xed\xf6\xef\xf7\x78\xd5\x11\x90\xec\xfd\x4b\x0f\x25\x6c\xf1\x89\xba\xee\xec\x50\x7e\x94\x5f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x97\x16\x8a\x9c\x3b\x07\xec\x49\x87\xa4\xcf\x1f\x24\x78\x73\x1f\xc6\x74\xf5\x6a\x2c\xae\xef\x07\x45\x90\xed\x6b"[..]);

        ctx.update(b"\x11\x19\xb9\x62\xbe\xd5\x81\x57\x34\xaf\x78\x27\xec\x53\x67\x01\xa4\x94\xac\x5d\x4a\xb8\x3e\xea\x1b\x16\xec\xc8\x0c\xe4\xe5\xf8\x69\x4a\x7d\x11\xbc\xba\x2e\x34\xf0\x84\xdd");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x20\x5d\x89\xe0\x32\xf0\x3c\x85\x19\xcf\x43\xb7\x20\x47\x83\x89\xb1\x78\x8f\x35\x22\xc3\xd3\x47\xfe\xbd\x2c\x70"[..]);

        ctx.update(b"\xd2\xc4\x5e\x2c\x1f\xa0\xc4\x4e\xfc\x84\xe6\xc0\x65\x4c\xc0\xd8\x67\xa3\xe3\x37\x33\xc7\x25\xaa\x71\x8d\x97\x4e\xd6\xa4\xb7\xf8\xf9\x1d\xe7\xd3\x62\x2b\x1e\x4b\xe4\x28\xde\x2a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd4\x83\xe3\x9b\x7a\xdd\x05\x0e\xb4\xa7\x93\xe5\x4c\x85\xb2\x50\x74\x6e\x38\x23\x99\xc7\x47\x36\xf3\x3d\xa8\x90"[..]);

        ctx.update(b"\xa8\x73\xb1\x48\xfe\x18\x07\xb8\x9c\xbe\xd9\x30\xa7\x80\x2a\xba\xd6\xca\x04\x42\x34\x0e\x62\xed\x21\xb8\x4e\xad\x9a\x63\x47\x13\xbb\x4d\xe5\x64\x82\x08\xc0\xee\xd6\x73\x8d\x9c\xc8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc8\x6b\xcc\x12\xa6\xab\x79\x2c\x14\x9a\xa8\x3a\x67\x83\xca\x8b\xb5\x2b\x0c\xa4\xb2\xc1\x26\x61\xc0\xa2\x5d\x22"[..]);

        ctx.update(b"\xb3\x00\x8f\x6f\x56\x7d\x1e\xed\x9a\xb5\xb3\xbb\xce\x82\x4d\x29\x0e\x66\xf6\x6b\xcf\xcf\xf7\xf9\xb8\x99\x48\x35\xb4\xd5\x4a\x4e\x45\xc9\xb8\x65\x1b\x37\xdb\xef\xe5\xe3\xfe\x5b\x67\x4f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x23\x92\x97\x53\xad\x07\xe8\x47\x6e\x7b\xda\xc8\xa0\xca\x39\xe9\xaa\xc1\x58\x13\x26\x53\xbe\x10\xeb\xee\xb5\x0c"[..]);

        ctx.update(b"\x78\xd0\x73\xb4\xe1\x3f\x68\x50\xdc\x1c\xa3\x66\x83\xab\xac\x72\x33\x64\x65\xd7\x90\xeb\x35\x75\xc9\x42\x66\x7d\x1e\x3e\xcc\x84\x9f\x37\xa8\xd7\x36\x04\xcb\x0f\xe7\x26\xff\xe5\x57\x44\xa2");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x62\x29\x23\x3f\xc6\x55\xea\x48\xbb\x5b\x48\xb7\x3a\x08\x18\x97\xd8\x55\xf6\xcf\x10\x47\x82\x28\xfc\x30\x58\x42"[..]);

        ctx.update(b"\x45\x32\x5b\x80\xe0\x43\xc0\xcd\xce\x3e\xc4\x21\xec\xda\x52\x94\x81\x91\x0c\x09\x73\x01\x28\xb4\xbb\x92\x7d\xda\x16\x59\xdd\xd8\xfd\x3c\xa6\x67\xd8\x57\x94\x1e\x6f\x9f\xd9\x39\xa1\xc5\x70\x98");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x77\x6a\xa1\xf5\x4e\x03\x8f\x39\x04\x91\xa5\xd6\x9b\xde\x7a\x2d\xbc\xba\x97\xc3\x55\x74\xeb\xe6\x0c\x9a\x77\x2f"[..]);

        ctx.update(b"\x3b\xdd\x68\x21\xd9\x38\xfa\xc5\x21\x01\xfb\xee\x5d\x6b\xa1\x91\xfb\x3b\x6c\xb6\x34\xdb\xf4\x2c\xeb\xaa\xe5\x7b\xd8\x97\x48\x1a\xe5\xee\x04\xe2\xd8\x71\xa4\xc3\x33\xab\x5a\xb6\x58\x81\x44\xf2\xf1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x62\xf8\xf3\xba\xea\x6d\xcf\x5a\xf2\x5d\x53\xdd\xfd\xac\x0b\xdc\xde\x88\xe3\x89\x5d\xf5\x67\xc6\xc4\x16\xa5\x41"[..]);

        ctx.update(b"\x86\xfc\x66\xf2\x61\x8c\x98\xfe\x9e\xfa\x1e\x3a\xc0\x4e\x34\x03\x85\xdc\x2b\x74\x6c\xbc\x0f\x7c\x75\x7b\x88\x34\x28\x10\xfe\x70\xd8\x12\x00\x95\x29\x28\xe7\xaa\xd0\xc0\xb6\xb1\x9a\x04\x45\x37\xb0\x09");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x20\xa2\x1e\xb1\xd3\x13\x0a\x45\x19\xce\x6a\xbd\x5a\xb6\x81\x70\x81\xae\x1b\xef\x36\x03\x05\x64\x76\xa0\x0e\x41"[..]);

        ctx.update(b"\xf2\xa6\x16\x8e\x7f\x92\xd3\x13\xfc\x30\xf9\xe6\xf8\x25\xa4\x80\x91\x62\x16\xf0\x2e\x03\x08\xdb\x70\x77\x3e\xc1\x65\xe2\x5e\x81\xff\xbf\x02\x20\xc5\xca\x0c\xc6\xc9\x1d\x3a\x09\xda\x99\xfa\x6e\xfa\x87\x7f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5d\x6e\x5c\x82\x57\x4f\x5e\x5c\x03\x39\xd3\xaf\x1f\x9c\x28\xe1\x7b\xcd\xdc\x30\x6a\x15\x18\x7a\xff\x5d\x3d\xd7"[..]);

        ctx.update(b"\x5e\x3b\x6b\x75\xb5\x4f\x21\xb8\x01\x6e\xff\xb3\x92\x76\xf5\xe7\xf4\x93\x11\x7a\xc4\xc0\xf2\xde\xc3\x8a\x80\xae\x29\x17\xda\xd8\x3c\x68\x90\x01\x20\xdb\x13\x25\xf1\xf4\x69\x7e\x0f\x5c\x25\xa8\xb9\x2a\x97\x02");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5d\xc2\x14\x7f\x1c\xf6\x55\xda\xbb\x5c\xa4\xb2\x97\x0b\x45\x64\xeb\x19\xec\x45\x6e\x6f\x96\x6b\xba\xe1\x97\x62"[..]);

        ctx.update(b"\xe7\xf1\x7c\x13\x19\x50\xc0\x63\x11\xf4\x77\x99\xa0\xf5\xa6\xb4\x99\x6f\x4c\xc8\x90\x33\x44\x50\xe1\xbd\x6c\xc6\xf5\x67\x07\x71\xc0\xdc\x60\x7f\x8e\xce\xb1\x53\x00\xec\x42\x20\x51\x0e\xd5\xb7\xde\xb3\x42\x9d\xe6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x4c\xe8\x0d\xab\x9f\x93\x31\x12\xa3\xfd\x78\xc1\xf7\x64\x34\xb1\x97\x80\x6e\xdd\xfe\x35\xcb\x0b\xdd\x84\x5c\x15"[..]);

        ctx.update(b"\xc9\xaa\x3d\x0f\x6d\x87\x8d\xb1\x12\x35\xe7\xb0\x28\xf8\xd6\x7e\x2c\xe2\x6e\xee\x71\x8f\x30\x8e\x21\x13\x2e\x37\x7e\x31\x70\xe2\x6e\xce\x95\xbd\x37\xa4\xbd\x7f\x87\x3b\xa7\xf8\xb7\x15\x17\xec\x50\x29\x7b\x21\xcf\x94");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x59\x63\xb4\x1b\x13\x92\x5a\x90\xc9\xe8\xfb\xcd\xed\x9a\x82\xad\xe8\xaa\xe3\x6d\xee\x92\x01\x99\xf6\xd6\xac\x7f"[..]);

        ctx.update(b"\x0f\x17\x0a\xfa\xfc\xef\xdf\xa8\xb0\xde\x32\x8d\xab\x30\xb4\xe4\x4d\x98\xd6\xae\xa2\xbc\x39\x55\x7f\xf4\x65\x8f\xce\x4f\xbf\x85\x26\xd8\xb5\x35\x9f\x17\x3c\x14\xe4\xda\x7c\xf8\x89\x35\xc9\x36\x9f\xc7\xd6\x07\x86\x3f\x25");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfe\x7e\x59\x02\x8c\x78\x55\xc3\x7a\xe3\xdc\x5e\xe3\x24\x86\x4c\xfe\xe6\xb8\xbc\xcc\x2c\x3b\x5a\x41\x0b\x65\xd9"[..]);

        ctx.update(b"\x6b\x2b\x92\x58\x41\x46\xa4\x33\xbe\xe8\xb9\x47\xcc\x1f\x35\xb6\x17\xb7\x3f\x5b\x1e\x03\x76\xac\x8b\xda\xdf\xe5\xbf\xdf\x22\x63\xb2\x05\xf7\x4d\xfa\x53\xdb\x7a\x29\xe5\x07\x8f\x5c\x34\xa2\x68\x11\x97\x36\xba\x39\x09\x61\xf6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x13\x2c\xfa\x7e\x71\xfe\x09\x91\xab\xbd\x88\xef\x58\x8a\xc9\x5a\xc9\x28\x9b\x1d\x77\x5b\x42\x03\x35\x67\xdd\x33"[..]);

        ctx.update(b"\x39\xf7\xa9\x43\x12\xbe\xa1\xb4\xfa\x98\x9f\x5a\x67\x75\xdf\x53\x8f\x01\x70\x41\x20\x83\x8c\x4a\x31\x04\x25\x64\x78\xb5\xc0\xcf\xbe\x8b\x86\xe2\x91\x2c\x98\x0b\x39\x0e\xa4\x12\xed\xdd\xb6\x9d\x46\x1e\x50\xf9\xf3\x13\xbc\x17\xaf");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfc\xc5\x96\x55\xb8\xfe\xc1\xa3\xd8\x78\x34\x5d\xf9\x10\x8b\xd9\x9f\x4d\xd0\xe5\x21\x8a\x55\xfc\x33\x5e\x57\xf7"[..]);

        ctx.update(b"\xac\x58\x2b\x5a\x4b\xb0\xc5\xe9\xc4\x0d\x8f\x27\x7b\xda\x9d\xe3\xd0\x7f\xff\x01\xe8\x20\xa1\xcd\xaf\x88\x70\x8f\x1d\x60\xbe\x60\xb9\xa5\xe8\x3b\x5c\x59\x36\x57\x38\x78\x02\xb4\x18\x2d\x1d\xf4\xe9\x46\x6e\x6d\x7a\xe6\xdc\x7c\x80\x79");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5c\x2e\x10\xfa\xe8\xf4\x30\x4c\xd9\x36\x16\x90\xe5\xd2\xc4\xcd\x15\xf1\x0a\x7b\x14\xea\x60\x20\x87\x39\x57\x9b"[..]);

        ctx.update(b"\x07\x27\x53\x98\x19\x98\x45\x34\x38\xa5\x20\xd9\xde\x2d\x57\x04\x29\x29\x10\x14\x8b\x8f\x79\x4e\xc3\x76\x5b\x24\x0c\x7a\xf1\xb7\x94\x62\xfa\x9a\x2f\x00\x0d\xd9\x4d\x59\x2d\x3a\x2a\x06\x9d\xc2\x44\xda\xf5\x7b\x12\xc5\x76\x75\xf3\xf8\x9b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb0\xd2\x90\xa6\xeb\xdd\x95\x08\x11\xa2\x71\x5f\x35\x4b\x0d\x89\x35\xcb\x61\x0a\x47\x1c\xfc\x5d\xff\x5e\x06\x60"[..]);

        ctx.update(b"\x66\xa9\xa6\xd0\xa3\x22\xed\x28\x52\x37\x8a\xf8\x2c\x0a\x2c\x02\x7b\x10\x82\x09\x8a\xb7\x50\x92\x5a\x4d\xc2\xe8\x96\x1d\x00\x62\xc9\xdb\x02\xe8\xcf\x42\xa6\xb4\x8a\xfb\x00\x56\xd6\xc1\xf1\xfb\xbe\xc3\xfb\xee\xf0\x49\x53\x5f\x6e\x9b\x38\x64");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd6\x83\x48\x8c\x84\x20\xeb\x2d\x61\xe5\x28\xab\x0a\x7b\x73\xaa\x78\x0a\x08\x5b\x9c\x79\x82\x29\x3b\x2a\xc6\xad"[..]);

        ctx.update(b"\x18\x41\x9a\x84\x98\xd4\xe9\xbf\xaa\x91\x17\x48\x18\x6c\x57\x53\xd5\xda\x5a\xa0\x33\x37\x1f\xfc\x56\x65\x0d\x0a\xe9\xb7\x3f\x43\x0f\x0d\x1f\x3c\x9d\x40\x36\x27\x86\xc0\x42\x9d\x97\x7b\x89\x9b\x64\x01\x6e\xca\x82\xe6\x42\x03\xf6\x68\x5c\x12\xee");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x51\xd0\xcd\x33\xfd\x65\x79\xb0\x5c\x36\x6c\x6f\xcc\x65\x36\x38\xb7\xb1\x3b\x62\x79\x8b\x99\xb3\x67\x92\xcd\xc4"[..]);

        ctx.update(b"\x4f\xc5\x20\x09\xd5\x8a\x0f\xc2\x57\x3e\x83\xfa\x33\x5b\x5c\x1d\xf8\xc1\x4b\x2e\x6d\xaa\xf0\x5b\xd6\xe1\x3f\xd5\x72\x2f\x28\xde\x48\x16\x77\x24\x24\xc2\xf9\x4d\xdc\x3d\xe0\xd3\xd7\xe2\x68\x12\xd0\x14\xbb\x9f\xd8\x30\x12\xdc\x9a\xbf\x1e\xc9\xe3\xf1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x63\x0e\xe2\xbe\xaf\x1c\x15\x92\xea\xa6\x26\x3f\xc5\x62\xa2\x60\xb6\x05\x4e\x9e\xab\x1a\xa1\x95\x36\xfd\xa1\x70"[..]);

        ctx.update(b"\xac\xda\xa2\x86\x92\xf3\x34\x73\x20\x88\xf5\xef\xab\x2c\x79\x51\xfe\x0f\x84\x5b\x9e\x2c\x6f\x12\x53\xc3\xcd\xcd\xe3\x0a\x4e\x8d\x21\x20\xe3\x8c\x26\x42\x22\x19\xdf\x41\xed\xa2\xc8\x33\x4e\x13\xf6\x69\xa6\x5f\x5b\xa2\x07\x5b\x46\x7e\xde\xd3\x29\x36\xd5");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x7d\x49\x91\xd5\x4c\x78\xaf\x58\x09\xcd\x17\x02\x4c\xad\xae\x78\x3c\x6f\x5a\x1f\x0f\xeb\x36\x5b\x53\x25\x80\xc2"[..]);

        ctx.update(b"\xd1\x59\x3c\xd3\x38\xb7\xa2\x5b\xb5\x41\x3f\x11\x2a\x63\x9f\xe3\x1c\x98\x1e\x50\x5c\x81\xa8\x20\xe6\x38\xc2\x52\x09\xe2\xce\x56\xc8\x83\x8a\x7c\x81\x17\xdb\xad\xcc\xde\xc9\x59\xa6\xf7\xca\xb0\xcf\x30\x43\x15\x70\x1d\x4c\xcf\x01\x67\xb4\x02\x6a\x67\x44\xde");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x84\xe1\x83\x30\x72\x3e\x4f\x90\x52\x0d\x0b\x05\x1a\x9b\xf9\xbd\x7b\x5c\x7e\xc0\x17\x78\x03\xf1\x5c\xf7\x40\xe5"[..]);

        ctx.update(b"\x8c\xf8\xea\x25\x31\x01\x26\xae\x1f\xdc\xe3\xc9\x19\x53\x95\xa9\xd4\x50\x51\xa2\xa3\xf0\x8c\xe1\x54\xd8\x26\x5b\x54\xcc\xa7\x03\x1a\x7e\xc8\x40\xc3\xa3\x35\x9e\xfa\x4c\x91\xc4\x1b\x74\xba\xa6\x98\xd5\x4f\xfb\x9b\x01\x70\xf2\xed\xad\xc5\x20\x16\x50\xc2\xbd\xc6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x75\xde\x14\x16\x9d\x16\xa9\x90\x2f\x6e\x8a\x33\x59\xd9\x45\x94\xa8\x89\xc4\xae\xd9\x24\x6c\xaa\x6c\xf5\x61\x2c"[..]);

        ctx.update(b"\xe0\x32\x0f\xee\x19\xaf\x5b\xfd\x51\x1a\x23\xca\xbb\xa7\x5a\xcb\x08\x15\x52\x5a\x37\x34\x30\x5a\xaf\xa4\x9c\x1d\x8b\xdf\xbd\x85\x35\x79\x64\x6a\x36\xa7\x87\x3c\x4c\xff\xf2\xea\xbd\x7e\x39\x02\xec\xcf\xf1\x19\x2a\xca\x1f\x6d\xce\x3c\xf1\xc9\x88\xe6\xac\xa9\xf2\xc8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd7\xf2\x01\x8c\x30\x3e\xe0\x45\xde\x4b\x8c\xde\xfc\xfb\x53\x95\x67\x4e\x3a\x87\x70\xd6\x5f\x07\x57\xb4\xcd\x5e"[..]);

        ctx.update(b"\x1a\x42\x4e\xcc\xe1\xa8\x2c\x47\x74\x21\x71\xa7\x01\xad\x6e\x0f\xf1\xa7\x62\xce\x26\xf8\xe3\x32\x81\x8a\x7f\xa1\xa8\x00\xa4\xe5\x06\xa4\xbd\xc8\x13\xa0\x9e\xe1\xd5\x72\x22\xad\xa7\x9a\x12\xe2\x39\x95\x49\xff\xd8\x0f\x16\x28\xef\x55\xe2\x31\xce\x09\x13\xf9\xab\x19\x30");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x27\x7f\x96\xfc\xa5\xd9\xab\x05\x5f\xae\x5d\x4d\xd1\x0c\xc4\x9c\x22\x37\xbd\x38\xd9\x5b\xd8\xdb\xd1\x68\xec\x21"[..]);

        ctx.update(b"\xaf\x17\x28\x09\x57\x0c\xc3\x06\x33\x3c\x25\x52\x3f\x86\x3c\x6d\x0e\x01\x54\xc5\x5e\x40\x47\x22\xf0\xd4\xed\x41\x97\x13\xda\xbf\x8e\x18\x49\x3a\x0e\x0b\x53\xb2\x20\xa3\x65\x35\xb1\xe8\xf0\xbb\xe4\x3e\x62\x4f\xac\x9f\x56\x6f\x99\x28\x07\xb6\xf2\xd7\x0b\xb8\x05\x93\x3e\x2e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x95\x81\x17\x00\x93\x60\x0c\xb6\x70\x63\xa3\x14\xd8\xde\xcf\x10\x9f\xf9\x36\x8f\xfb\xc9\x0e\xa2\xd3\x25\x05\x77"[..]);

        ctx.update(b"\xa6\x2f\x4b\x43\x25\x0c\xdf\x3f\x43\xc1\xda\x43\x9b\xc5\xe4\x22\x4b\x15\x18\x5b\x60\xd6\x15\xe3\x8e\x3c\x51\x24\x25\xaa\xb1\x45\x40\x1b\x57\xac\x3f\xc0\xbc\xc1\x78\xea\xfe\xf5\x2a\x2b\x7b\x04\xb2\xb8\x9e\x76\x02\x12\xf9\x6c\x4e\xe6\x94\x99\x08\x31\x85\x8f\x0f\xa7\xc1\x3c\x24");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa0\xf5\x77\x5a\x2d\x00\x1a\x66\xf0\x88\x2c\xe1\x41\x52\x61\x99\x40\x21\x98\x86\x90\x84\x0c\x6b\x4a\x34\x70\xc8"[..]);

        ctx.update(b"\xfc\xf8\x1c\x93\xf9\x17\xbb\x06\xf2\x78\xf4\x88\x26\xef\x9c\xa8\xba\x99\xac\x8f\x00\x12\x9f\xd9\xf8\xe8\x1c\xa3\x17\x50\xd5\xe5\x48\x18\xaf\x03\x31\xdd\x23\x9e\xb7\x7e\xe4\xb0\xc4\xd0\xc2\xd8\x47\x94\xce\xf2\x7d\xa6\xbf\xeb\x70\x77\x94\xd3\xbd\xbc\x7b\x34\x99\x68\xf2\xa3\x16\xd8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa9\x7a\x74\xfb\x01\xfe\xc5\xca\xf3\x47\x72\x20\xee\xf6\xe7\xc3\x6d\x0b\xa4\x19\x9d\xdc\x75\x5f\x7c\xcf\x94\xee"[..]);

        ctx.update(b"\xe6\x1d\x24\xb5\x00\x58\x17\x34\xc2\x99\x02\xad\xe4\xc5\x03\x5c\x09\x08\x68\xdf\x9f\x24\xbb\x33\x06\x09\xfc\xdf\xf4\xa7\x2d\x6f\x18\x00\x14\x24\xfd\x81\x3c\xea\x32\x92\x3d\x8a\xa8\x6c\x3d\x21\x5b\x2a\xb7\xd1\x34\x23\x7b\xb6\x2e\x78\xf6\x1c\xb9\xe9\xb4\xef\x5c\xed\x23\x72\x9d\x01\x9a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x40\x75\x83\x14\xf1\xab\xbd\x43\xe0\xbc\x9c\x73\xa1\xc7\xe2\x47\x19\xd5\x6e\xeb\xcd\x96\x7b\x39\xd3\x55\xe9\x78"[..]);

        ctx.update(b"\x37\xb1\x4f\x04\x23\x3d\xfb\x4d\xa5\xe5\xbd\x18\x52\xf7\x7c\x41\xe2\x5c\x49\x26\x93\x6f\xe4\x14\xc8\x10\x82\x00\xf6\xf3\xcd\x78\xc0\x3e\x2d\xd9\x61\x54\x46\xc1\x4b\xeb\xc2\xc7\x0d\x65\x50\x6a\x7a\x5d\xec\x48\x08\x80\x62\x91\x76\x9e\x0d\xba\xb2\x00\xe5\x76\xf9\xfd\xb9\xe2\x40\xc8\xb8\xff");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x2d\x36\xaf\x0d\xd9\x56\x19\xa9\x6c\x56\x64\xd8\x98\x7b\xbb\x82\xd1\x83\x46\x6f\xf4\x41\x51\x03\x4f\xed\x68\x7b"[..]);

        ctx.update(b"\x45\xef\xb0\xa3\xd8\xfb\x7b\xb6\x83\x91\x34\x59\x72\x7e\x87\x56\xd6\x79\x59\xcf\xdd\x4f\x5b\x80\xe1\x3d\xdf\x45\xe0\x9d\xeb\xdc\x2c\xc6\x8c\xeb\x63\x2d\x6d\x45\xa2\xd0\xa8\x69\xf6\xd4\xdc\x4c\x13\x6c\x80\x58\x49\xfe\x77\xb4\xb3\x81\xe4\xc6\xb2\x2a\x3f\xf6\x99\x47\xa9\xb5\xaa\x6b\x7c\xbe\x42");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x12\x5e\x98\x32\x29\xf6\x5b\xf0\x1b\x59\xa9\xb6\x19\x81\x0a\x88\xf1\xc5\x3b\x4c\x3b\x19\x60\xb5\x2a\x20\x5d\x99"[..]);

        ctx.update(b"\x9b\x6c\x3c\x77\x74\x62\x19\xdd\x88\x97\x69\x66\xc6\x8e\xad\x59\xeb\x62\xaa\x3c\xf6\x64\x77\x98\xdc\x06\xd4\xfc\x7e\xf8\xbd\x44\xd8\x90\x3f\x1b\x7b\x6f\x8b\xbf\x3d\x62\x49\x05\x2f\x86\x2e\x9c\xcf\xb0\xd1\x95\x7f\x0b\xba\x23\x36\x03\xbc\xa0\x76\x62\x86\xd1\x7e\xb9\x74\x6b\xc0\x02\xab\xd6\x95\x83");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x76\x26\x29\x51\x88\x33\xba\x68\x33\x3f\xc3\xe3\xb4\xd4\x82\xc6\x0b\x4e\x0e\x82\x88\x72\x82\x6b\x68\x31\x33\x15"[..]);

        ctx.update(b"\x9f\x45\x2f\x90\x02\x19\x01\x71\x99\xed\xfc\x5d\x7d\x86\xa1\x62\xd9\x75\x0b\xba\x4c\xec\x77\x42\x8e\xd1\x03\x2e\x57\x11\xb6\xfb\x7c\x37\xc1\xa6\x5b\x3d\x04\x1c\x7a\xa1\xd4\xf1\x6b\xbc\xfc\x54\xf3\x50\x01\x43\x6b\x60\xab\xfb\x65\x44\xc0\xb3\x93\xfc\x13\x89\xe5\xc5\xbd\xbd\xf2\xea\xab\x1d\x99\xdd\x59");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x19\xb4\x32\xf5\xc3\x8f\x66\x54\x41\xd3\x6c\x47\x2d\x38\x60\x08\xa5\xbb\xd8\x2a\xa4\xea\xbe\xaa\xbe\x3d\x28\xcc"[..]);

        ctx.update(b"\xcb\xfd\x18\x65\x92\xfa\x68\xdc\x3a\x21\xd6\x2d\xb1\xba\x55\x12\x1f\x58\xfe\xcb\x11\x69\x58\x59\xd7\x0b\xd7\xed\x2a\x21\xa2\xa0\x13\xa6\x99\x64\x08\x42\x97\x3b\x57\x1b\xf4\xa7\xc8\xee\x4f\x61\x7d\x5e\x8a\x4d\x1e\x8c\x15\xae\x33\xe7\x70\x97\xd1\x46\xeb\xa2\x79\x34\xb1\xe3\x3d\x8a\x04\x1f\x24\x44\xca\x3a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb3\x2a\xd1\x3b\xa4\xa0\xb9\xfc\x1a\xa9\xa1\xa5\x7b\xdb\xfb\xeb\xdf\xab\x71\xcf\x5a\x16\xe0\x60\x40\xf7\x57\x87"[..]);

        ctx.update(b"\x17\x32\x25\x32\x4c\x6c\x35\x0d\xdb\xa2\x27\xb8\x9a\x65\x1e\x57\x6d\x1a\xb6\xa9\x68\x95\x45\x3c\x33\xea\x61\xdd\xb3\x7f\xa2\x53\xe6\x66\xa8\x4d\x0f\xea\x60\x98\x14\x68\x84\x95\x24\x61\x61\xeb\x9c\xcc\xdd\x79\x2c\xb1\xb8\x8f\x36\xf3\x12\x5d\x76\x6e\x2e\xab\xe8\x41\x75\xcb\xe6\x6d\xbe\xcc\x91\xa0\xcc\xf1\x73");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfc\x8f\xee\xca\xef\xff\xda\xa9\x66\xe9\x53\x6b\x91\xdf\xc8\x5e\xa5\x11\x3a\x01\xd6\xb3\x20\x67\x7d\x72\x7a\x7d"[..]);

        ctx.update(b"\x69\x99\xf3\x98\x40\x74\x80\xcd\x43\xba\xfd\xae\xdb\x86\x24\xd9\xba\x09\x72\xaa\x5a\x2f\x35\x04\xa6\x7f\xe5\x4e\xf7\x44\xb7\xbb\x41\xea\x70\xcf\x8f\xaa\x77\x1f\xac\x6a\x2f\x58\x23\xde\x83\x82\x6a\xf4\xc3\x86\x5b\x6f\xae\xee\xe3\xd1\xd0\xed\xfe\x7f\x0e\x9f\xe3\x20\x7f\x91\x7b\x46\x7d\x84\x18\x50\xfc\x6e\x64\x8f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe7\xab\xcb\x4c\x0f\x21\x88\x14\xec\xf4\x5f\xbf\x28\xa3\xf2\x86\xd9\x0c\x5e\x74\x0a\xaf\xd1\x64\x74\x37\xc1\xe1"[..]);

        ctx.update(b"\x27\x27\xee\xb1\xd5\x10\x98\xc6\x9f\xd8\x14\x1d\x78\xf2\x12\x75\xb2\xbb\x94\x9e\x71\x15\xfd\x38\x60\x52\x6b\xbd\xa2\x55\x47\xc2\x0c\xf3\x1b\x79\x91\x9f\xa3\x7b\xfd\x47\x26\xc4\xe7\x79\x06\xff\xe0\xca\x97\x05\xf1\x78\x2d\xa0\x45\x4e\x79\x94\x22\xc8\x15\xe0\x1e\x78\x5d\x41\x8f\xa8\x81\xf8\x43\x41\xd8\xcd\x71\xec\x77");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x2b\xe3\x32\xc8\x73\xed\x4f\xb7\x0b\xc1\x91\x6c\x76\xbe\xf2\xcd\x33\x85\xe6\x74\xb8\x3a\xa1\xee\x8a\xd2\x8a\x01"[..]);

        ctx.update(b"\x1f\x48\xa5\xb4\x01\xd8\x8e\x6c\xbe\x37\xf3\xf6\x34\xd5\x54\x62\x86\x5f\x7c\xde\x79\x90\x05\x2a\x1e\x4a\x1e\x4c\xb2\xe5\x8c\x84\xc2\xc7\xef\x82\x92\x34\x47\xd7\xc0\x68\xb6\xaa\x25\xe3\x88\xac\xfc\x05\x70\x4e\x46\xda\x14\x31\x6d\x37\xcc\xdd\x27\x06\xa7\xb7\x9d\xde\xb0\x2d\xcd\xd7\x6f\x34\x2c\x9c\xb2\xf4\x90\xc1\x8d\xc1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x44\x8b\x70\xf5\x75\xa8\xa1\xeb\x74\x03\x0a\x98\x5e\x9c\x50\x4d\x4e\xaf\x6b\x18\x14\xe1\x14\x6f\x78\x2c\x9a\xf5"[..]);

        ctx.update(b"\x6d\xce\x9a\x9e\xcb\x48\xb9\xda\x8a\xef\x51\xa8\x9e\x7f\x7f\xc1\xa6\xa7\x89\x66\xb7\xba\xc0\xac\x5b\xa7\xab\x18\xd9\x2b\x61\x6b\xb7\x45\x37\xbf\x7e\xeb\x9b\xd3\xbd\xfb\x40\xa4\x50\x74\x7c\x3d\xe2\xe6\xee\xcf\xb1\x27\x63\x04\x91\x48\xfa\x91\x34\xc7\x87\x0b\xa8\x06\x36\xfb\x21\xfc\x71\x34\xf9\x2b\x03\x64\xf5\xd2\x7d\xea\xca");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xdf\x85\x5d\x54\x4e\x17\xf0\x11\x25\x02\x2b\xc1\x8e\x9f\xfc\xed\x12\xf3\xcd\x39\x67\x4e\x68\x18\x46\x57\xec\x7e"[..]);

        ctx.update(b"\xd4\x98\xb6\x90\x13\x45\xaf\xdd\xc5\xaa\x50\xca\xc7\x7f\x7f\x79\x4d\x79\x29\xee\xd5\x71\xd9\x5b\x59\xc2\x89\xa0\xc9\xf3\xb8\x12\xb8\x96\xbc\x7b\x56\x6f\x5a\x63\x9e\xd9\x94\x8e\xd0\x66\xc2\xc6\x22\xc6\xe4\xdb\xb2\xea\x37\xe7\xc0\x68\x06\xd6\x1a\x22\xc3\x26\xd7\x23\x56\xec\x48\xc9\xb5\x18\x2c\x29\xb5\xf9\x23\xaf\x20\x04\x66\x05");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5b\x22\x5c\x29\xe4\x54\x77\x77\xa2\xc6\xa1\xa2\xbb\xe9\xda\x2d\xc6\xa8\xc6\xd0\xd0\x1d\x8d\x80\x22\x98\x8b\xe2"[..]);

        ctx.update(b"\xe9\x58\xb8\x04\x89\xaa\x6a\x38\x52\x62\x44\xda\x16\x5d\xc4\x46\x4e\x79\x61\xe4\x57\xf7\x63\xab\xdb\x23\xf7\xe4\x8d\x36\x83\x31\x19\x7b\x37\xcd\x5a\xb1\xe5\x15\xce\xb1\x12\x48\x48\x50\x4d\x8b\xe5\x87\xbf\x30\x41\xd1\x04\x37\xeb\xd5\x39\x15\x16\x45\x56\xb5\x91\x06\xbe\xbd\xf9\x91\x15\x12\x2d\x99\x52\x9e\x02\xee\x15\x51\x38\xa1\x3a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x36\x4a\x98\x84\x00\x42\x45\x57\xa9\xc6\x0e\x4e\x1f\x32\xf0\x85\x5a\x33\x83\xc9\x0b\x00\x7d\x30\xee\x3e\xc3\x33"[..]);

        ctx.update(b"\xf3\x3b\xa9\x82\xbc\x2c\x33\x08\xf9\x48\xa1\xb6\x4c\x7f\xb6\x8f\xb8\x91\xbc\x05\xfa\x18\x78\x1b\x1d\xc9\x5d\xc7\x49\xf7\x00\x9a\xdc\x58\xcc\xa2\xbb\x0c\xf7\x90\xeb\xdb\xb4\x16\x5b\xbf\xab\x93\x04\xa2\xa6\xf2\x34\x68\x8d\xcf\x27\x30\x94\xdc\xd8\xd7\xb3\x84\x16\xbe\x57\xce\xda\xce\x57\x83\xd8\xb9\x29\x93\x54\x82\x56\xb5\x37\x3f\x2b\x4e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xca\x37\xe5\x2f\x28\x43\xa0\xf6\x56\x92\xc5\xae\xed\x01\x69\x60\x1d\xa3\x27\x5d\xfb\x3e\xe6\xd8\x1b\x46\x7f\x60"[..]);

        ctx.update(b"\x8b\x5d\x77\xa9\x06\xc7\xec\x75\x63\xaf\x75\x51\xa7\x96\xe5\xd5\xdc\xf0\x2c\x42\x12\x1d\x7b\x13\xa4\x9a\xa9\xd4\xbc\x79\xd6\x37\x19\x0e\x4e\x65\x10\xec\xaf\x92\xd1\x10\x4f\xd4\xec\x5b\xd8\x35\x14\x46\x35\x07\x22\xd1\xb2\x77\x5d\xbc\x5e\x65\xf8\xfa\xb4\x73\xdc\x63\x7b\x5c\xa8\xa9\xeb\x88\xf6\x8d\x11\xdd\xe1\x52\x75\xd7\xc4\x72\xf9\xdb\x43");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x93\x37\x53\x7d\xe4\x82\xf0\xcf\x88\xca\xd6\xb8\x6e\x19\x5a\x1e\x42\x2e\x59\xcc\x60\xd4\x1d\x0e\xca\x8b\x00\x91"[..]);

        ctx.update(b"\x3a\x56\x4a\x84\xc2\xb4\x8e\xe2\x6d\xa1\x38\xce\x2d\x1a\xe3\xc7\x93\x3b\xcd\x65\xe4\x02\x88\x40\x6e\x56\xf3\x0d\x1c\x48\x69\x0a\x49\x98\x38\x9d\xd2\x7b\x55\x37\x6f\x9b\x4e\x7f\x43\x60\x7f\xad\xb1\x6e\x89\x33\x72\x6f\x00\xa3\xe4\x12\x64\xcd\xa5\x53\x53\x27\x61\xfe\xfc\x73\xe8\x6e\xd7\x9b\x84\x9b\x94\xe0\x89\x54\x51\x33\x2d\xc8\x0f\xe3\x9a\x4b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x88\xea\xb3\xe1\x6c\xa8\xda\x57\x16\x54\x2b\xae\x3c\x7c\x73\x6b\x54\x1c\x89\x61\x99\xb2\xcb\x94\x12\x13\x76\x7b"[..]);

        ctx.update(b"\x61\x8a\x53\x98\x9f\xfb\xbf\x54\xa7\x6f\x01\xf9\xb8\x77\x72\x49\x1d\x87\xc8\xf2\x5c\x58\xeb\x11\xb1\x8a\x04\xf5\xba\x8e\xd6\x25\x74\xc3\x51\xa4\x66\xdf\x64\x73\x1c\x91\x14\x58\xd7\x65\xcb\xde\x83\xe7\xf2\x9d\xe9\x0b\xc1\xbb\x26\xcc\x56\xb3\x5c\x14\x05\x55\xa7\xdc\xf0\x0f\x53\x94\xd7\x6a\x4c\xc5\x31\xd7\xd5\xf5\x7b\xac\x7d\xcb\xd0\x6a\x4f\x73\xba");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x4a\x72\x7c\xc6\xb4\xbd\x93\xd5\xff\x2e\xcb\x81\xab\x50\x57\xdf\xdc\xbe\x3e\x0c\x49\x43\x6a\x58\xb9\xff\x3e\xf2"[..]);

        ctx.update(b"\x31\x85\x7b\xb4\xe8\x24\x97\xb5\x26\xe4\x26\xde\x69\x20\xa6\x06\x3d\x02\x26\x4d\x52\x49\xfe\xff\xd1\x4a\xbd\xbb\xf0\x35\x63\xd4\xc5\x9a\xd1\xf7\x57\x2c\x7d\x0e\xfb\xc4\x6a\x65\xde\xa9\x58\x0b\xde\x0e\x38\x7c\x9e\xdc\xe2\x7c\xd9\xb2\x0a\x46\xf6\x2a\x70\xe6\xdd\x5f\x58\xe4\x0a\xac\x3a\x22\xdf\xb6\xba\x07\x3f\xac\xda\xdd\x58\xcd\x6f\x78\xc0\x2b\xd2\x19");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9e\x61\x4f\xc1\x39\x64\x5e\x15\x8c\xd1\xb2\x16\xe2\x62\x3e\x58\x62\x42\xaf\x64\xf8\x48\x3e\x6f\xca\x20\xed\x4b"[..]);

        ctx.update(b"\x14\x85\x90\x08\xc8\x3f\x28\x31\xbe\x4d\x6e\x54\xb7\x81\xb9\xfb\x61\xda\xdc\x40\xc4\x59\xa9\x3e\xde\x11\xb4\xc7\x8a\x7e\x5a\x55\xa7\x17\x01\x42\x75\x26\xa0\x3b\x42\xd8\x83\xf2\x47\x90\x48\x13\xcd\x81\x2e\x7a\x94\x7c\x8f\xa3\x74\x06\xaa\x61\x45\xae\xa6\xd3\xfd\x9e\xd4\x94\x18\x6f\x35\x33\x3d\x42\x3c\xe3\x1e\x0c\xd4\x73\xa0\x31\xa5\x80\x3c\x55\x93\xe9\xa4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x54\x5f\xaf\xa4\x3a\xfc\xaf\x38\x06\x3d\x8a\x31\x2c\x3a\x27\xe0\xd7\x4b\xff\x95\x7f\x8e\xf4\xd5\x1c\xb2\x96\x98"[..]);

        ctx.update(b"\x26\x7a\x14\xba\xd7\x02\xef\x0a\x84\x68\xb3\x1c\x72\x71\x5f\x05\x33\xf6\xb9\x7e\x6e\x94\x38\x39\xde\xa4\x20\x71\x9d\x6d\xef\xc5\xa3\x99\xf8\x46\x89\xe6\x4e\xcf\x93\x1e\xe3\x95\xee\x49\xf1\xfe\x36\x21\x99\xb7\x3c\xc6\xcb\x01\x05\xb3\x65\x4b\x16\xf1\x9f\x06\xee\x8a\xa6\xb5\xd5\x41\x87\x43\xd4\x80\x4f\x9a\x05\x92\x70\x71\x0d\x12\x67\x65\xe6\xa4\x9c\x4c\xe2\xe3");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9b\x93\x60\xa5\xc7\x47\xe6\xe1\x28\x8f\x6f\x9d\x97\x10\x51\xff\xd8\x46\x41\xf6\xd6\x4e\x0a\x4b\x51\x42\xe4\xec"[..]);

        ctx.update(b"\x6c\x98\xa8\xeb\x3e\xa4\x45\x14\x01\xe0\x42\x4c\x10\xcb\x72\x26\x83\xb2\x3f\x75\xae\x25\x4d\x62\xeb\xa7\x5a\xbb\x9a\xa9\x69\x8e\x65\xba\x1f\xf7\xc9\xf8\x6d\x36\xd1\xca\x6f\x04\x25\xd1\x94\x28\x44\x1b\x00\x45\x0e\x9a\x2e\xf6\x85\xd5\xda\x1c\xd4\xde\x1e\x77\x91\x84\xdb\x74\x3f\xc9\x5a\x46\x17\x97\x33\x38\x08\xae\x6e\x42\xfc\xe1\xe9\xda\x5d\x82\xf9\x0c\xd7\x1b\x54");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x0c\x6f\x33\xf9\x53\x4f\xc5\x2f\x37\x00\xf3\x7b\x9e\xe6\x78\xb4\xc5\xc8\xa9\x0b\x1a\x2e\xb1\x57\x40\x02\xe3\x77"[..]);

        ctx.update(b"\x4b\xae\x62\xa0\x08\xd9\xfd\xba\x35\x1a\x19\x03\xc6\x6d\x58\xe5\x87\x36\x19\x90\xf7\xc9\xee\xa0\x5a\x2f\x51\xf9\x0a\x28\x92\xf6\x0e\x6c\x14\xc4\xed\x36\xb9\x08\xc4\x03\x9b\xc8\x97\x97\xfd\x88\xe5\x42\x81\xb3\x7f\x61\x9b\x3d\x9a\x27\x45\x87\x22\x9e\xf4\x83\x51\xe8\xcb\x18\x81\xcb\x0f\xc8\x3e\x6d\xdc\x90\xa0\x5b\x16\x0f\xd7\xd0\xa1\xeb\x08\x35\xd5\x71\x58\xe4\x2c\x7b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x98\x9c\x15\x6b\xa1\xfd\x1f\x70\xde\xb3\x78\xe4\x6f\xfc\xbf\x6f\x2c\xf9\xcf\x97\x7a\x92\xac\x51\x64\x3c\x97\xb4"[..]);

        ctx.update(b"\x83\xca\x6d\x4e\xbd\xf1\xc0\x40\x62\xca\x1a\xbb\x97\x76\x70\xef\x9b\xcc\x88\x99\x06\x93\x5f\xd6\x4f\xf4\xc7\x39\x91\x2e\x54\x1b\x8f\x8c\x79\x32\xf5\x95\xef\x66\xe1\x82\x56\xdf\xa1\xf5\x1f\x63\xbf\xe7\xa9\xdf\x3a\xe2\xaa\x43\x17\x71\xd1\x93\x18\xd6\xaa\x3b\xcc\xfa\xc1\xa4\xc8\xaa\x0a\x04\x33\xff\x80\x7a\x88\x1e\x0d\x5a\x97\x22\xaa\xc6\xcd\x57\xc7\x7e\xb6\xa9\xed\xf8\xc0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfb\x83\x1f\x24\x56\x59\x5f\xab\xee\x9d\x45\x86\x25\x28\x3a\x80\xbb\x4f\x8f\x03\x1e\x9a\xbd\xbf\x48\xb7\xb5\x1e"[..]);

        ctx.update(b"\xf4\xc7\xad\x8d\x24\xed\x5a\x68\x2c\x47\x34\x63\xe8\x53\x91\x05\x0c\x02\x6f\xef\x0b\x0e\x6d\xca\x38\x8e\x1a\x7e\x2b\xc8\x72\xa4\x67\x46\xa6\x3f\x3a\x2c\x1c\xa6\xe4\xc8\xb7\xc5\xfb\x6b\x58\x85\x0d\x77\xa5\x89\x88\xba\x09\x1b\xd7\xfa\xfb\x66\xce\xd1\x84\xe5\x48\xbc\xfb\x1b\x0e\x6e\x14\x85\xfb\x6a\x19\xcd\x5e\xd0\x76\x40\xa0\x77\x7b\x82\x27\x3d\x5e\x80\x79\x9b\x7f\xa7\xa5\x7d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x13\xbe\xe6\x17\x47\x4b\x3f\xc3\x44\x70\x25\xf2\xa4\x88\xdb\xa8\x82\x5d\x46\xa4\xe1\x28\xb9\xa8\xbd\xeb\x1b\x85"[..]);

        ctx.update(b"\x5f\x81\xc5\xae\xc9\x23\x85\xbf\xdc\x55\xeb\xd6\x00\xf2\x3c\xb0\x4a\xc9\xd5\xc7\xa1\x39\x6f\x80\x1f\xfe\xa1\xa6\xb9\x4a\xa6\x17\x23\x17\x61\xbd\xee\xbc\x9e\xc0\xf4\xbf\x9b\xfa\xf5\xeb\xc7\xac\x82\xa2\xc9\x6f\x1a\x74\xc4\x6d\x94\xf0\xda\xd0\xbc\xb9\xef\x7b\x41\xdd\xaf\xf8\xcf\x63\xd2\xb2\x78\x23\x9e\x65\x58\xdb\xae\xd2\x79\x7e\xf3\xb7\xf4\xcf\xf8\xfe\x59\x2f\x6a\x35\x51\xb3\xd7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x14\x3a\x6f\x0a\x20\xd5\xb4\xdb\xc5\xdf\x64\xa7\xe5\x0f\x99\x85\x63\x14\x53\xeb\x09\xde\xd7\x16\x67\x70\x90\x83"[..]);

        ctx.update(b"\x07\x35\xce\xca\xed\xef\x99\xbf\x4c\x53\x24\x2f\x05\x52\xf4\x9f\x56\xbb\xe5\x89\xa2\xf6\x11\xaf\x75\xf4\xf3\xae\xc3\x66\xcd\xd6\x70\x2d\x46\x39\x15\x12\x58\x02\x02\xb8\x69\x09\x7f\xce\xb8\xa4\x58\x89\xfb\xbf\x98\x52\x47\x2f\x94\xbc\x2f\x43\x2b\xb8\x30\x9c\x4d\x0c\x4d\x3f\xba\x01\xf6\xe9\x0c\x5c\x2e\xa3\xf8\x90\xed\x95\xd1\x32\xc3\x1f\x4d\xad\xbf\x26\x8c\x37\x8f\xac\x56\x04\xe8\xa4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9f\x5e\x9f\x74\x29\xe5\x48\x8a\x84\x3c\x52\xff\xb4\x6a\xe2\xe8\x42\x28\x91\x9d\x32\x33\x0a\x91\x93\xaf\x3b\x21"[..]);

        ctx.update(b"\x9b\x4e\x4d\xf9\x2e\x51\x52\xfe\x1e\xc5\x6a\x9f\xc8\x65\xf3\x0b\xac\x7e\x94\x9f\xc4\xf6\x2f\x0b\x15\x8d\x10\xb0\x83\x63\x6b\x4d\xe9\xbb\x05\xdb\x69\xfe\x31\xb5\x01\x03\xfe\xfc\x5f\x8d\xaf\x3a\xf7\x15\x6b\x45\x52\xca\x36\x67\xa9\xd7\x20\xbb\xb2\xe4\xbc\xda\xba\xdf\xd4\xb7\xf4\xfc\x5b\xc8\x11\xfa\xa3\x67\x10\xa9\xd1\x77\x58\xa9\x8d\x4a\x04\x74\xfe\xc2\x7e\x9e\xf5\xb7\x4f\x5c\x68\x99\x35");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x48\x7a\x6f\x2f\x87\x5c\xb2\x53\xde\x4c\xef\x18\xec\xb4\xf2\xa5\x43\x88\xeb\xaf\xfb\xfc\x42\x59\xbd\xd9\x7f\x09"[..]);

        ctx.update(b"\xa6\x1b\xef\x83\x88\x67\x71\x0f\xf4\x34\x1b\x26\xb1\x3b\x8d\x7a\xf7\xe4\x61\xcc\xd3\x17\xb1\x60\xcc\x4f\xda\xae\xc7\xf1\x80\x5a\x28\xdd\xd3\x66\x3a\x42\x10\xa7\xd1\xb6\x4a\x75\x2e\x86\x6a\xa7\x22\x4a\x75\xbf\x77\xbd\x0d\x61\x8b\xcc\x3b\x0a\x3e\xed\x6b\xfe\x0e\xb2\xb8\x82\x81\x9e\x6a\x4c\xc4\x37\xbd\x38\x91\x5c\xe5\x3c\x55\xd9\x4e\x9e\x93\x39\x28\x64\x83\xdc\x23\x0d\x00\x49\x77\x7e\xa1\xc4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe2\x57\xbc\x45\xb6\x2d\x08\x53\xba\x4b\x0f\x85\x78\x69\x8f\x42\x62\xc3\x1a\x77\x8c\xb6\xa6\x31\x7b\x6e\x6d\x60"[..]);

        ctx.update(b"\xc0\xbd\x79\xe0\xc5\xf7\x2f\xcb\x1d\xe6\xc2\x34\xbd\xb6\x7b\xd0\xd3\xf4\x81\xb9\x62\xa3\xa0\x1f\x2d\x8c\x48\x3b\xd7\xd5\xd9\x85\x48\xd5\x1d\x27\x53\x27\x16\xb1\x95\xfd\xfb\x0e\xa0\xb7\x7d\xb7\x59\xb5\x4e\x26\x9e\x69\xe4\x8e\x2c\xb0\x7b\xc9\xc0\x62\x59\x92\x7d\x27\x55\xf4\x8e\x8d\x9a\x02\x0c\x58\xa9\xc9\x22\x1a\x9d\x83\x6f\x03\xb3\x0e\xab\xf9\x09\x9c\x8e\xeb\xa6\xab\xed\x63\xbb\x38\x27\x5b\x28");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x92\xdf\x7f\x84\x8a\xda\x8a\x96\x98\xdd\xc2\xe7\x45\x2a\xc8\xfc\x43\xcf\x83\xd2\xca\x2c\xad\xd7\x12\xc5\x95\xf2"[..]);

        ctx.update(b"\x77\x82\x3a\xf9\xb8\x79\x6c\x63\xba\xeb\xe7\xba\x9d\xcd\xe1\x2c\x62\x6b\x84\x0e\xa0\x4f\x42\xd8\x78\x64\x69\x70\xca\x5b\xf7\xab\xa9\x4e\xaf\x11\x0d\xa3\x6c\xe0\xc8\x34\xb6\x54\xbc\xac\x93\x26\x4a\x34\x9f\x52\x0e\x50\x5f\x1e\xc9\x03\xd3\x58\x9e\x3a\x4a\xdf\x82\x68\x7a\x65\xee\x6d\xd0\x72\xd6\xbc\x05\xac\xdf\xbd\xf2\x57\xcd\x70\xa5\x18\x3a\x54\xb4\xfe\x8e\x87\xd1\xc2\x2b\x2e\x9f\x4e\xe8\x17\xc5\x7d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x81\x9a\x43\x40\x93\x84\x97\xcd\x8b\x1d\xef\x84\x44\xbb\x03\xf8\x42\x9b\x9e\x87\xba\xd8\x00\x00\x02\xd6\x0b\x83"[..]);

        ctx.update(b"\xad\xa5\x65\x1b\x4e\x24\x03\x35\x60\x09\x40\xf2\x07\xb9\x83\x71\xf7\xe7\x43\x98\x89\x57\xbf\xfe\x0d\xe8\xef\x08\x62\xd1\xba\x52\xc5\x2b\x69\x50\xe7\xb0\x5c\x35\x42\xc2\xfb\x13\xac\xaf\xf0\x44\x2d\x33\x94\x0a\x0e\x3e\xa6\x72\x32\xf8\x43\x7e\xaa\x02\x12\x82\x83\xff\xc0\xcf\xe2\x54\xac\x8f\x54\x2b\xe3\xf0\x5f\xbe\x4e\x85\x5d\xd2\x2a\xe9\x8a\x81\xb9\xa5\x5b\x3d\x37\x53\x11\x12\x10\x04\x8f\x2b\x50\xe0\x68");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb6\x17\x7d\x17\x9c\xf1\x7e\xdd\xcd\x89\x88\xc9\x10\x8b\x42\xaf\x9c\x41\xad\xcc\x59\x42\xc4\xd3\x3b\x0f\x1b\xe2"[..]);

        ctx.update(b"\xff\x47\x04\xbb\xbd\x71\x9b\x01\x12\x44\xeb\xed\xf2\xf2\x35\x53\x38\xfc\xc7\xd6\x48\x44\xc3\xa0\xf3\x6a\x21\x56\x9b\x55\xf7\x4a\x97\x10\xf8\xf3\xd8\xd8\x3b\x9b\xcd\x73\x3f\x58\x85\xc3\x2b\x3d\x14\x9a\x5a\xd1\x37\xd0\x16\xc0\x3b\x93\xa4\xd1\x1a\xff\x82\x18\xe8\xee\xec\x6d\x6d\x12\xa4\x1d\x14\x41\xf3\xdf\x04\x0f\xeb\x09\x8c\xa2\xf0\x03\xc4\xc2\x77\xfc\x71\x30\x0c\xdd\x2a\x39\x9a\x7b\xb9\x8a\xe7\x11\xc4\x46");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa1\x07\x2b\x28\xf3\x45\x34\x22\xe6\x11\x42\x13\x09\xaa\x49\xaa\xeb\xba\x02\x73\xc7\x2b\x83\x5f\xde\xea\x11\x32"[..]);

        ctx.update(b"\xea\xe4\xb6\x2f\x69\x7c\xf0\xbf\x40\xa1\xc2\xc1\x09\x14\x3c\x1d\xde\x18\xe2\x4f\x1c\x28\x9a\xba\x67\xe5\xc8\x3e\xef\x52\xb7\x0c\xf1\x43\x3b\xb9\x80\x13\x94\x92\x85\x96\x96\x30\x05\x4e\x07\x4c\xa2\xe2\x49\xd4\x65\xcb\x38\x3d\xba\x51\x56\x1c\xbc\xb6\x26\xf0\xb3\xb1\xd5\x42\xdb\x1e\x1f\xf1\x68\xf3\x71\xc7\xc6\x76\x4b\x4f\x25\xad\xe9\xeb\x35\x16\x22\x21\x2e\x99\x90\x36\x14\xbb\xf1\xfe\x39\x14\xcd\xf2\x03\x03\x5a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf5\x27\x3e\x4d\x0b\xf9\x77\x9a\x09\x75\xfe\xe2\x3c\x44\x7b\x3a\xbb\x1c\xd1\x7c\x34\xc7\x23\xd6\x2f\x3a\x2f\xd1"[..]);

        ctx.update(b"\x0e\x39\xe0\xe6\x93\x3c\x61\x04\x98\x4f\xff\xe1\x15\xdd\x8c\xde\x77\xed\xfe\xe4\x95\x48\x0a\xa5\xe5\xde\xf4\x24\xf0\x66\xa5\x77\x03\x45\xfe\xcb\x28\xb1\x6c\xaa\x54\x16\xbc\x79\xe2\xb8\x31\x45\x40\x9b\xd4\xbf\xe9\xa0\x0c\x84\x93\xf0\x6e\xa2\xa9\x9d\xd6\x58\xfb\x87\xb7\x1e\xb5\x7d\xaf\xe5\x8d\xa5\x5f\xa0\x41\x1e\x79\x03\x41\xe3\x1a\x8b\xa8\xf3\x5b\xbe\x71\xaf\x23\xb4\xe8\x83\x3f\xd6\x5e\xc8\xb4\xe6\x21\xe9\x53\x40");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x62\xfb\x7d\x6b\x38\x10\xd0\xfd\x7d\x96\xb4\xff\x5e\xfe\x7b\xd2\x83\xdd\xbb\xed\xa4\xa2\x1a\x62\xf9\x85\xa3\xdc"[..]);

        ctx.update(b"\xe3\x2b\xea\x9a\xb0\x2d\xe7\xd8\x93\xec\xb7\x85\x7b\xa6\x6d\xf2\xc3\x5e\xd2\x58\x12\x30\x65\xca\x80\xe2\xa0\x67\xfa\xbb\x7d\xd4\xe7\x98\x39\xea\x0b\x3c\x58\xab\xac\xe8\xe9\x7b\xf4\x2b\x0b\x8d\x97\xfc\xb0\x9b\xb6\x06\xa1\xda\x02\x43\xc3\x2d\x24\xcc\x98\x98\x5d\xf0\x08\xf8\x69\x83\x62\xf2\xaa\x78\x9e\x2a\x82\xb3\xe5\xb5\x01\x18\x53\xd0\xc0\xe8\xfb\xd2\x0c\x4d\x2b\x5f\x47\x33\xf2\xdf\x8c\x5a\xe0\x2e\x92\xa9\x0d\x95\xd3");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x27\x8e\x06\xfd\x12\xa3\xe3\x14\xf6\x0d\x59\xa3\x23\x67\x3b\xa0\xa2\x20\x03\xe4\x2a\xc4\x8e\x1c\xd0\x4a\x70\xd0"[..]);

        ctx.update(b"\x41\x57\x75\x2d\x3d\x17\x5a\x4b\xc1\x33\x4f\xd4\x2c\x20\x41\x11\x72\x8e\x70\x59\x65\x9d\xce\xdf\x33\x4e\xa7\xce\x30\x37\x87\x98\xd6\x7c\x59\x8a\x0a\xfa\xcc\xa5\xa1\xc5\xfb\xa9\x23\xd5\x4c\x72\xcf\xfc\x98\x87\xdf\x1b\x8d\xf1\x0d\x96\x51\x49\x55\x05\x68\x15\xfd\x2d\xd8\x55\xd3\x2e\x8b\x58\xb6\xfd\xf4\xd4\x57\x15\xf6\x36\x41\x6a\x01\x37\x17\x9f\x7e\xb0\x1d\x78\x6d\xaf\xfa\x92\x4c\xca\xbd\x52\x3b\xb3\x1d\x1b\x5f\x0d\x05\xc4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x1c\xab\x43\x63\x5d\x50\x1e\x43\xac\x42\xbe\xee\x26\x37\x55\xb9\xa2\x98\x27\xe2\xa1\x8b\x21\xd7\xbe\x42\xe4\x47"[..]);

        ctx.update(b"\x2d\xf1\x2d\x8c\x25\x6c\xd1\xa1\x27\xe5\x25\xac\x37\x63\xe3\x0c\x89\x59\x82\xee\xe6\x7a\xb7\xc1\x50\xce\x3d\xea\xe9\x06\xd2\xb9\x11\x0d\x82\x9c\xcf\xdf\x27\x93\x72\x9e\x31\xe4\x78\xe3\xa3\x10\xae\x52\x5e\x05\x99\x71\xa2\x95\x15\xba\xd2\x27\x3c\xee\x77\xad\x89\xad\x88\xd6\x3d\x44\xe9\x84\x02\xc6\x31\x80\xcf\x5e\xb0\x6d\x0b\xe3\xb1\xfa\xf5\xad\xfc\x5c\x43\xa7\x9f\xfc\x09\xa6\xee\x6c\xdd\xf9\xc9\xa0\x39\x42\x1d\x5b\x21\x84\xad");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xee\x60\xf0\xd0\x10\x08\xcf\xac\xe4\x9a\xf2\xee\x57\x80\xcc\xde\xe3\x74\x04\xc3\x76\x42\x00\x8a\x55\xfa\xfa\xf2"[..]);

        ctx.update(b"\x03\xbe\x69\x40\xe8\x59\xf9\xb0\x72\x66\x0d\xff\x28\xa1\x87\x55\x1c\x24\x25\x48\x1d\xd0\x55\x5d\x2d\xee\x4a\xcc\x36\x16\x4f\x84\xf8\x50\x5b\x6f\x46\x7a\xe6\xf7\x72\xea\xfc\xc9\x06\x54\x90\xd9\xb4\xed\x12\xa6\x90\xd0\x44\xbf\x7d\xa1\x49\x86\xe5\x71\xfe\x34\xae\xe2\x8e\x1d\x69\x8c\x41\x36\xcc\x9f\x95\xd4\x62\xc9\x90\xb6\x81\x5a\x54\x46\x7d\xa6\xf4\x1c\x1b\xaa\x86\xc4\x48\xf3\x7a\xc1\x0b\xbc\x2a\xd1\xb9\x57\xb1\x73\x68\xce\x01\xa7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa8\xaa\x80\xd4\xc9\x25\x88\x9b\x58\xef\xf4\x1b\x89\x68\x2b\x92\xbe\xa6\x0c\x1c\x39\x95\x04\x3d\xac\x31\x2d\x2d"[..]);

        ctx.update(b"\x0b\xaf\x1a\xc2\x43\xc1\xf3\x4c\xa5\xe0\x0a\xed\x4d\x86\x7f\x96\x7b\xc2\xb9\x63\xe9\x39\x56\xc3\x5b\x6b\x68\xda\x77\x37\xde\x23\xd7\xa1\x40\x5a\x5d\xd4\xa0\x99\xc6\x63\xcd\xc1\x82\xd4\xc9\x1b\xc3\x5f\x7d\x3f\xd5\xf3\xac\x35\xad\x7a\x26\xdb\xc4\x5e\x3e\x86\x26\x4c\x7d\xec\xc5\x38\x98\x42\x14\xa1\xa0\xa1\xd1\x16\x79\xae\x22\xf9\x8d\x7a\xe4\x83\xc1\xa7\x40\x08\xa9\xcd\x7f\x7c\xf7\x1b\x1f\x37\x3a\x42\x26\xf5\xc5\x8e\xb6\x21\xec\x56\xe2");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf1\x2f\x7a\x1c\x5c\x1c\x38\x3a\x2a\x5f\xff\x89\x32\xe2\xae\x9d\xc3\x42\xb3\x76\x52\xd4\x73\x56\xff\xc1\xcb\x37"[..]);

        ctx.update(b"\x3c\x29\xa8\xc8\x3e\x48\x19\x4a\x7b\x87\xb6\x9e\x37\x6a\x06\x06\x3d\xe2\x44\x9b\xd1\x71\xfa\x91\xe5\x8e\xd2\xbc\x90\x4b\xa8\x53\xbb\x35\xe3\xf5\x1e\x7c\x06\xe9\x6b\x54\x82\xaa\xc8\x9a\xcf\xa3\x83\xbb\xba\x37\x01\xd2\x01\x04\xf8\x10\x1d\x69\xde\x61\x5f\x45\xa2\x4c\x3e\x02\x99\x1b\xf0\xd3\xbb\x3d\x37\x39\x0f\xe8\x7e\xcc\x64\x03\x24\x38\x42\x42\x18\x86\x20\x93\xa6\x9d\xd7\xb9\x90\x08\x57\x36\x61\xf9\x99\x6f\xfe\x8e\xd5\x0b\x7e\x54\xf4\x9c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5c\x6b\x29\xc3\xcb\xfd\x1d\x2e\xad\xf7\xc7\x91\x51\x3b\x27\xf2\x1c\x93\x4d\xe6\x37\x8e\xf7\x48\xb7\x79\xb7\x1d"[..]);

        ctx.update(b"\x68\xa3\xc0\x6e\x07\x40\xb5\x69\xc7\x2e\xa6\xa9\x0d\x8b\x45\xe8\x3c\x7c\x35\x0d\x2b\xcf\x1c\xf6\xd6\xdf\xfa\x75\x53\xb8\xb9\x98\x08\x7c\x05\x2e\x1c\x06\x5d\x86\x2b\xcc\x6a\x7a\x3e\x0a\x90\xac\xfa\x1d\xc4\x10\x17\x2c\x9d\xab\x14\x0e\xad\x9a\x29\x68\x11\x55\x7e\x16\x47\x35\x9a\xcd\x40\x34\x1e\xfe\xb6\xf5\xb3\xfd\xc0\x04\x41\x62\xa4\x5e\x62\xb0\xec\x34\x16\x34\xbc\xec\xb8\x30\x62\x69\x30\x39\x2f\x8c\x6b\xde\x85\xfa\x08\x8a\x32\x20\x54\xac\xfc");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x58\xa6\x91\x52\x43\x98\xa5\x74\x6d\xf2\x8a\xc0\x83\xf1\x58\x61\x75\x0e\x0c\xdd\x1f\xd5\xe5\xf5\x7c\x98\x2c\x18"[..]);

        ctx.update(b"\xd4\xf7\x57\xd1\xc3\x3b\x9c\x0b\x38\xb4\xe9\x3e\x8e\x24\x83\xec\x51\xb4\x86\x12\x99\xf1\xd6\x50\x96\x14\x57\x49\x6d\x86\x61\x4d\x42\xa3\x6e\x36\x96\xbf\x16\x8f\xd4\x66\x3e\xfc\x26\xe8\x8c\xd5\x8d\x15\x1e\x15\x31\x46\x7b\x73\xf6\x9d\xc9\xce\x4f\x8d\x41\xce\x57\x9c\xe1\xc9\x1e\x67\x60\xe3\x40\xe7\x67\x7a\xbd\xf4\xfe\xc1\x04\x07\x45\xaa\x51\x44\x64\x0a\x39\xb8\xc4\xf8\x84\xdf\x80\x75\x3a\x69\x16\x53\x00\x3d\x63\x4f\xa5\xbf\xce\x81\xf9\x4e\xc3\xf6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xbe\x11\x25\x93\x77\xf0\x98\x21\xd9\xdc\x35\x85\x92\xb6\x56\x5d\x8e\xf2\xb4\x14\xdf\xaa\x7d\xb5\x60\x9f\xb7\x51"[..]);

        ctx.update(b"\xec\xd9\xe9\x5f\x7c\x5e\xfc\x83\x36\xf8\x0f\xe6\x7e\x11\x36\x57\xb3\x14\x82\xba\xfc\x22\xdc\x5b\x45\x07\x34\x82\x84\x6c\xdc\x48\x41\x4d\x2e\xa8\x55\xae\x75\xd9\xf2\x8a\x0b\xdb\xe3\x0d\xbe\x51\x15\x03\x78\x8e\x57\x8f\x20\xf2\x5e\x20\xbb\x77\x0c\xa1\xd7\x87\xf2\xf0\x29\x11\x13\x92\x75\xdb\xea\xa5\xae\x1a\xaf\x15\x5f\x40\xd7\x13\x49\x15\xda\xc3\x4d\x09\x38\x35\x8d\xc8\xbe\x97\xcf\x10\x05\xa9\x22\xbf\x3d\x71\xc3\x31\x28\x2f\x41\xc8\x69\x93\xe0\xcc\xff");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x69\x50\xad\x0f\x91\x39\x8b\x39\x96\x5b\x18\x59\xea\x91\x8c\x53\x12\x12\xfa\xce\x1e\x51\xd4\xd3\x90\xf0\x94\xe1"[..]);

        ctx.update(b"\x83\x4d\xdd\x8f\xc7\xea\x0c\x33\x85\xef\x82\x80\xd3\xa7\xb2\x2d\x59\xad\x17\xd7\x10\xa5\x1a\x54\x4a\x29\x35\x44\xf3\x06\x59\xe8\x16\xa9\x8d\x38\xa2\xd4\xd9\x2f\x6f\x96\x62\x6a\x7c\x79\xd6\xf1\x7b\xfd\x0a\x55\x8f\x45\xe2\xfb\x54\x11\x72\xb7\x20\xec\x62\x9c\x88\xa7\x97\x13\x26\x05\x0f\x2b\x9a\xb8\x0d\x30\xcf\x8c\x77\x7f\x80\xe3\x7c\x98\xfa\x61\x79\x75\x23\xe8\x1e\x1b\xbb\xc7\xcd\x6e\xe2\x2e\x42\x49\xda\xe6\x79\xce\x0f\x3e\xcc\xfb\x54\x49\x5d\x7e\x70\x46");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xef\x21\xee\x8d\x56\x8c\x00\x9e\xaa\x8d\x1e\xa7\x70\x96\x8c\xb7\x18\xc4\xd5\x6e\x7b\x2d\x96\x6b\xfc\xbb\xf3\x98"[..]);

        ctx.update(b"\x6f\xf6\x11\x20\x83\x95\xd8\x15\x00\x50\x5d\xae\x05\x0f\xf0\xc2\x9c\x0a\xfd\xe2\xa8\xe8\x9c\x96\x19\x28\x63\xea\x62\xc1\x7e\x29\x2d\x05\x02\xe9\x4d\xcb\x7f\x47\xf4\xcd\xd5\x74\x26\x4f\x48\x71\x6d\x02\xd6\x16\xcf\x27\xc7\x59\xfd\xf7\x87\xcd\xcd\x43\xb1\x69\xea\x58\x6c\x8b\xca\x25\xfa\x3c\xe1\xa0\x8e\xb6\x15\x65\x5e\x24\x71\xa0\xfa\xa8\x1d\x2e\xdc\xa2\x8e\xff\x40\x30\xfa\xbf\x36\xf1\x0f\xb5\xf5\x0f\xe4\xeb\x72\x7c\x30\x8f\x31\x7b\xba\x99\x5b\x63\x10\xae\x12");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8a\x29\xf2\xc0\xd5\x64\x93\x5b\x8d\x31\xb7\xd0\x07\xf5\x81\x38\x48\x9d\x14\x09\x17\xa2\x8e\xe8\x5d\x43\xb6\xf2"[..]);

        ctx.update(b"\xf9\x77\xea\x38\x07\x63\x28\xbb\x0e\xe2\x29\x7c\xbe\x3b\x2a\x97\x55\xfe\x8b\xb9\x5a\xe7\x26\x29\x8e\x04\xdf\x05\x20\x1a\x7c\xcf\x20\x46\xb8\x28\x36\xe0\x92\xda\x94\xa4\xeb\x1c\x29\x14\x50\x12\x17\x18\x15\x94\x68\xe8\xa3\x30\xfc\x2b\x12\x72\xc6\x61\xfb\x62\x39\x7e\x87\x4f\xfc\xd7\xcc\xcb\xe5\x42\x5a\xf7\x25\x79\x10\x01\xc0\xc0\x35\xea\x41\xc8\xc4\x8d\xab\xd2\x06\xdd\xb2\x17\x66\x6e\x2b\x68\x82\x37\xc2\x12\x7e\x96\xeb\x04\x9d\x94\x1b\x34\x12\x6b\x37\x3e\x13\x45");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x15\x18\x0d\xf5\x55\x43\x87\x33\x7f\x04\xde\x2f\x37\xa1\x6b\x28\x12\x5a\xdb\xd0\x2b\x6f\xa6\xcf\xdb\x24\x19\x5d"[..]);

        ctx.update(b"\x22\xa8\xfb\x43\xd5\x4f\xff\x82\x74\x9c\xdc\xe9\x8a\xbe\x8a\xda\xfc\xd4\x43\xff\xe1\x6b\xf0\xe9\x93\x41\xe1\xf7\x06\x4f\xc0\x7a\x59\x07\xc8\x16\xab\xdb\x32\x6c\x30\xfe\xf0\xf5\x84\x6e\x9e\x31\x3f\x32\xb6\x02\xc9\xe0\x03\x52\x70\x63\x58\xfc\xb7\xfb\x81\xea\xf1\x85\x7a\x7b\x0f\xfd\xdf\x27\xb7\x41\xa4\x65\x96\x18\x06\xcc\xf6\x72\xc1\x79\x93\xf2\x84\xb2\xaa\xa9\xa2\xc8\x54\x25\x0a\x42\x12\xaa\x79\x37\xa9\xbf\xee\xfc\x30\xec\x5f\x00\x67\xc3\xaa\xf3\x4a\x1d\xce\x2e\xe6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd1\x1f\xcb\xbb\x2f\xa0\x31\x09\xf9\x52\xa5\x6e\x16\x86\x7c\x70\x90\x45\x52\xeb\x58\x0a\x66\x59\x31\x4b\xd5\xfe"[..]);

        ctx.update(b"\x68\x72\x76\x36\xff\x38\xc0\xba\x89\x99\xdd\xe3\xcb\xd9\x50\x39\x00\xd5\xcc\xb0\x1d\x3c\x9b\x79\x59\xfb\x41\x1e\xed\xf9\x5c\xce\x18\x05\xce\xf6\x67\x0d\x1e\x11\x33\x90\x1c\xc0\x6b\x55\xc4\x1d\x94\x5e\x65\x4c\x0d\x18\x03\x54\x98\xd4\xf9\x2d\x16\x7a\xe2\x1b\x92\x7c\xba\x3a\x81\x0a\x41\x59\x48\x85\xa0\x0b\xff\x35\x4f\xfc\x75\x3e\x36\x82\x74\xd0\x13\x74\x46\x9f\x1b\x3f\x77\x93\xe4\x36\xdd\xc0\x82\x2a\xd6\x98\xf1\x3b\xd1\x5f\xb3\xed\x10\xe0\xb9\x7f\xac\x5f\x87\x78\xd9\xce");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x21\xc7\x1b\xd0\x9e\xbf\x5d\x09\x15\x53\x47\xc4\xf4\x76\xb8\xf9\xc5\xae\xd4\x57\x95\x73\x21\x18\x87\xab\x60\x84"[..]);

        ctx.update(b"\x16\x7c\xb7\x72\xf0\x96\xb2\xe3\xb1\x59\x9c\xce\x34\x40\xd1\xaf\x57\xc5\xb7\xdf\x5d\x2f\x46\x0b\x91\xac\xc7\xe5\x2c\x9f\xdb\x19\x79\x3b\xc0\x83\x37\x51\xd0\x9f\x3f\x66\x4a\x41\x67\x09\x55\x86\xa5\x64\x42\x0a\x78\x10\x12\x5b\x83\x2e\x38\xae\x7b\xb3\xa0\xd1\x44\x03\xef\x61\x57\xc2\x0d\x3d\x67\xe6\xe1\x3a\x44\x11\x5b\x19\xff\x1f\xb8\xb6\x4f\xfa\x01\x81\x33\xb6\xd5\x32\xd9\xda\x69\xb9\xbf\xfb\xcd\x74\x18\x90\x71\xa5\x71\x01\xe7\x23\x94\x01\xea\x50\xad\x1e\xa0\x4a\xab\x96\x1c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc4\x6c\xb2\xdf\xeb\x8b\x96\x1e\x6e\x84\xd7\x2e\x05\x11\x1e\x04\xd6\x2e\x3f\x93\xa0\x55\x16\x4b\x13\x5b\x90\x72"[..]);

        ctx.update(b"\xb8\x8f\xf7\x28\xc8\xf8\x29\x84\x1a\x14\xe5\x61\x94\xbb\xf2\x78\xd6\x9f\x88\x31\x7a\x81\xb4\x74\x9a\xa5\xfd\xbc\x93\x83\x48\x6e\x09\xbf\xf9\x6a\x2c\x5b\x5b\xdf\x39\x2c\x42\x63\x43\x8a\xef\x43\x33\x4c\x33\x17\x0e\xf4\xd8\x9a\x76\x26\x3c\xb9\x74\x5f\x3f\xea\x74\xe3\x5f\xbf\x91\xf7\x22\xbb\x13\x51\xb5\x64\x36\xcd\xd2\x99\x2e\x61\xe6\x26\x67\x53\x74\x96\x11\xa9\xb4\x49\xdc\xe2\x81\xc6\x00\xe3\x72\x51\x81\x34\x46\xc1\xb1\x6c\x85\x8c\xf6\xea\x64\x24\xcd\xc6\xe9\x86\x0f\x07\x51\x0f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x88\x91\xcd\xfe\x48\x6a\x58\x2e\x83\x40\xbd\x8b\x89\x39\x96\xd7\xa4\xe5\x47\xe3\xbf\x50\x55\x19\x02\xe7\x22\xf2"[..]);

        ctx.update(b"\x52\x0f\x27\xa4\xd0\x96\xd4\x19\x3d\x2b\xc0\x98\x3c\xf8\x3b\xbb\x50\x84\x84\x5b\x41\x84\x48\x00\xc1\xf5\x66\x9b\x4f\x67\xf5\x78\x5c\x9c\x88\x6e\xac\x51\xb0\x59\x00\x5c\xc3\xca\xf2\xf7\xdc\xfc\x20\x5c\x23\x0a\x8c\x92\x4f\x60\x43\x86\x69\x6f\x3d\x5d\xd2\xa6\x85\x09\x87\x9d\x99\x1a\xa4\x93\x14\xd7\x27\x1a\x8a\x8e\xf7\x11\xb4\x28\x25\xd3\xcd\x00\x71\xae\x3b\xf6\x10\x97\x72\xbf\xac\x1b\x16\x7f\xad\x99\x5f\x99\xb7\xaf\xc2\xc5\x73\xf2\xce\x64\x93\xe2\x54\x11\x10\x1d\xca\x79\xb6\xd2\xf1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x21\x6e\xa5\x09\x97\x59\x6f\x71\xed\xc9\x4e\xd9\x6e\x2b\x68\x66\x28\x64\x0f\x94\xa3\xc6\x4a\xde\xf0\x5c\x2b\x63"[..]);

        ctx.update(b"\x75\xc2\x3e\x55\x61\x78\xf0\x04\x40\x53\x3b\xcd\x25\x25\x79\x34\xd0\xc6\xf5\xe6\x8a\x64\xf1\xaa\x51\x1b\xee\x94\x35\xc5\x27\x7b\x02\x14\x5f\xae\x1f\xde\xdc\xe3\xb6\xb7\xb4\x70\x15\xc5\x47\xbe\x55\xd0\x0d\xfa\x39\x99\x92\x0d\x58\x6d\xbe\xcf\x7f\xf9\x5a\x77\x51\x60\xd0\x57\x30\x8b\x32\xc6\x61\xc1\x7e\x5d\x6a\x77\x21\x66\xbf\x69\xb9\x91\x9e\xe9\x1f\xe9\x38\x77\xa5\x07\x11\x93\x9c\x85\xa9\xcf\x1a\xb6\x5c\x28\xfa\x94\x87\x96\x23\xfa\xec\xe2\x0e\x14\x58\xb8\x82\x13\x83\xfd\xa2\x25\x37\x62");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd1\x63\x10\x28\xa8\xe0\xec\x4a\xdc\x68\x9c\xab\xba\x8b\xf6\x81\xd1\x1e\x2e\x2a\x50\x59\xf2\x93\xf7\xef\x5b\xe3"[..]);

        ctx.update(b"\xd2\x33\x73\xb9\x40\x50\x24\xd0\xc4\xb1\x7a\xa5\x03\xf7\xe2\xff\x7d\x30\x80\x83\x12\x4e\xd2\xcb\xc4\xd9\x90\xb9\xbe\xe0\xd7\x0b\x96\x35\x87\x2f\xcf\xda\xea\x58\xa2\xb6\x96\xd1\xfd\x8c\x94\x92\xcd\x2e\xc1\x11\x79\xee\x75\x5a\xae\x56\x63\x62\x62\x19\xc0\x98\x13\x48\xa8\xbe\x50\xc9\xbd\xf7\x7b\x06\x11\x21\xcd\xe2\x46\x64\x9a\xf1\xf3\x0b\xd7\xe8\x4a\x93\xd9\x52\xf8\x02\x5f\x85\x4d\x7b\xd3\xd5\x9d\x0e\xcd\x07\xe6\xd4\xd9\x09\xb2\x3c\x7a\xe0\x3f\xa0\x6f\xe1\xde\x1c\x34\x24\x99\x9f\xcc\x36\x18");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x72\x6f\x65\x84\xff\x9e\xa9\x98\xff\x32\x6c\x9f\x73\x29\x1a\xce\x87\x26\xd8\x69\x7e\x7a\xa9\x4f\x1e\xd4\x2f\x7e"[..]);

        ctx.update(b"\x6f\x05\x7f\x91\x48\x0f\xec\xee\x8a\x7e\x38\x79\xdb\xf8\xc5\x20\x40\xf9\x6f\x59\x29\xc6\xb8\xb6\xae\xa2\x23\xb9\x18\x43\xdd\xeb\xa3\x87\xa2\x28\x82\x64\xdf\x3d\x24\x1d\x14\xb5\xb6\xbc\x7d\xef\xe9\xbc\xf1\x74\xf5\x06\x0a\x88\xde\x1f\x86\xff\xf5\x9f\xed\x52\xa3\xe5\x74\xf2\x62\x09\x22\xdc\x0c\x12\x31\x6e\x58\x69\xb7\x79\xa1\x8e\x86\x97\xea\x0a\x50\xbf\x20\xa5\x0f\x16\x9e\xd8\xa3\x08\xf7\x85\xbd\x98\xef\xe6\xfd\xf4\xca\xc4\x57\x4d\xca\xe9\xbb\xe5\xf3\xd7\xf5\x6a\x11\xba\xd2\x82\xfc\x9c\x84\xa7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6b\x40\xe5\xc8\x6d\xb3\xd9\xc3\x84\xc2\x2a\x46\xcb\xef\x5f\x8e\x8c\x42\x7b\xb6\xbf\x43\x26\x8e\xdd\x91\x8a\xeb"[..]);

        ctx.update(b"\x6f\x77\x87\x4d\xca\xd9\x47\x9f\x5b\xca\xc3\x76\x36\x62\xcc\x30\xcb\x99\x82\x3c\x5f\xf4\x69\xdc\xbd\x64\xc0\x28\x28\x6b\x0e\x57\x95\x80\xfd\x3a\x17\xb5\x6b\x09\x9b\x97\xbf\x62\xd5\x55\x79\x8f\x7a\x25\x0e\x08\xb0\xe4\xf2\x38\xc3\xfc\xf6\x84\x19\x8b\xd4\x8a\x68\xc2\x08\xa6\x26\x8b\xe2\xbb\x41\x6e\xda\x30\x11\xb5\x23\x38\x8b\xce\x83\x57\xb7\xf2\x61\x22\x64\x04\x20\x46\x1a\xbc\xab\xcb\x50\x04\x51\x9a\xdf\xa2\xd4\x3d\xb7\x18\xbc\xe7\xd0\xc8\xf1\xb4\x64\x5c\x89\x31\x5c\x65\xdf\x1f\x08\x42\xe5\x74\x12");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x02\x28\x62\x6c\x63\xc2\x04\x65\xd5\x13\x9d\x1a\xf0\xb9\xce\x17\xe3\x34\xeb\xe1\x0a\x5e\xee\x2c\xaf\xe9\x6c\xb1"[..]);

        ctx.update(b"\xea\x84\x1b\xd4\x1b\x22\xe4\xc9\x8b\x22\x33\x32\x91\x8e\xb7\x91\xf5\x1d\x19\x78\x54\x07\x85\xf9\xc6\x17\x67\x5d\xbd\x02\x72\x18\x31\xf7\xe7\xfd\xfa\x77\x14\xaf\x7d\x67\x1b\x58\x8a\x64\xf4\x9d\x85\x56\xb5\xd1\xc4\x48\x11\x68\x39\x77\x1f\xaf\x51\xa8\x5d\xbb\x1b\xbf\xf5\x9f\xad\x8e\x3f\xe3\xc4\xeb\x86\x31\xaa\x05\x0f\x50\x5d\xf8\x57\x57\xed\x9e\x9d\x1a\x26\xa8\xa0\xe9\x6f\xee\xaa\x7a\xf2\x04\xcd\x23\xfd\x0e\x6d\x4c\xa8\xd5\xff\x25\xb9\x1a\x0f\x94\xc4\x2a\x88\x72\x97\xb2\x30\xf6\xd5\xd5\x72\x71\xe0\x7c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xff\x33\xc6\x42\x31\xde\xdf\xc2\x47\xe1\x1e\x35\xaa\xf8\x2d\x28\x3a\x9a\xd6\x20\x34\x10\x2e\xe2\xbb\x5d\x46\x09"[..]);

        ctx.update(b"\x72\x16\xa8\x25\x02\x9d\xa1\xc9\xa9\x32\x8d\x49\x9b\x3f\xf9\x8f\x6e\x18\xb8\xaf\x36\x8e\x2b\x19\xef\xc1\xc0\x12\x1b\x35\xb9\x65\xab\x28\x2f\x55\x23\x23\x56\xd7\xfa\xd0\x02\xfe\x3f\x0b\x6a\xb7\x83\x3b\x2c\xb6\xf2\xe3\x92\xb0\xc3\x74\x14\xcb\xd3\x66\x1e\x53\x8c\x86\x13\xae\x0c\x92\x91\x92\x83\x03\xf7\x75\xdd\x2a\x24\x45\xa2\x7e\x82\x5a\x1a\x35\x44\xa9\xb4\x11\xeb\x3a\xa8\x7d\x0f\xdc\xdc\xd8\x5c\x17\x05\x11\xdb\x62\x0e\x74\x72\x96\xbd\xc3\xaf\xa3\x94\x89\xc1\x81\xf5\xab\xc7\x6a\x8a\x40\x4e\x47\xe4\xa2\x14");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x94\x40\xd3\x71\x0b\x43\xe7\x98\x99\xe1\x16\x98\x73\x66\xb2\xdd\x36\xb4\x4b\x2f\x39\xe3\x77\xfa\x2d\x4f\xe1\x43"[..]);

        ctx.update(b"\x44\xa8\x50\x8a\x3c\x39\x76\xd5\x63\xe9\x33\x70\x5b\xe4\xdb\xee\xbc\x72\x63\x04\xb5\x11\x20\x3d\xf7\xc7\xd1\xef\xce\xb6\xe0\x6e\x91\xf1\xe5\x7f\x3d\x8e\x6c\x10\x5d\xfd\xf8\x26\x2d\x98\x48\x16\xfe\x7a\xd8\xf8\xdc\x95\xab\x59\x6f\xff\x48\x30\x1f\x8d\x03\x13\x7b\xa3\x7d\xab\xdc\x4a\x6e\x66\x45\x83\xa2\x6b\x8e\xdc\x42\xd3\xc2\x40\x55\x16\xc5\x13\x86\xc3\x3a\x7f\x28\x75\xa3\x08\x77\x02\xca\x67\x21\xf5\x61\x95\x05\x3f\xe5\x26\x3a\x29\xc8\xd8\x53\x8d\xce\x6c\xe1\x46\xb8\xb4\x3a\xe5\x20\xee\x79\xa5\xa4\x50\xc6\xa2");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa2\x74\x3d\x34\x10\x23\xff\x5f\x77\x5d\x90\x18\x5d\x31\x39\xa7\x75\x6b\x0a\x65\xc1\x9e\xe8\x76\xeb\xeb\x92\xae"[..]);

        ctx.update(b"\xa8\xef\x41\x07\xf4\x1e\xbb\xc5\x79\x9a\x71\x6b\x6b\x50\xe8\x7c\x19\xe9\x76\x04\x2a\xfc\xa7\x70\x26\x82\xe0\xa2\x39\x8b\x42\x45\x34\x30\xd1\x5e\xd5\xc9\xd6\x24\x48\x60\x82\x12\xed\x65\xd3\x3a\x5c\xa2\xbc\xdc\xa7\x72\x80\x37\xdf\x2e\x5f\x9f\xd9\xe9\x74\xd0\x31\x5d\xde\x82\x90\x24\x1e\x3e\x2b\x2c\xc0\x6f\x8c\x65\x3e\xbc\x95\xbc\x21\x95\xc2\x4d\x69\x0c\xae\xd4\x2f\xe7\xd9\x65\x89\xf3\xa8\x5e\xae\x9b\xad\x99\x5a\xb8\x29\xe6\x74\xab\xcf\xb8\xef\xaa\xcb\x1e\xee\x57\x03\xf5\x2b\x97\x9d\x5d\x99\xa1\xc1\x69\x48\x55\xa0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb4\x11\xa2\x8f\xf4\x65\x13\xd0\xc3\xd6\x3c\xf7\x8a\x9b\x63\x53\x46\x6c\xba\x3b\x92\x6a\x8d\x89\x5e\xe1\x4f\xdd"[..]);

        ctx.update(b"\xf6\x49\xd8\x01\xb4\x04\x0b\x7b\x51\x52\xf5\x8a\x01\xe7\x85\x2f\x56\x5e\xfc\x77\xb5\xda\xfe\x46\x07\xee\xe9\x53\xb0\xba\x67\x74\xc5\x57\x3f\x1c\x79\x76\x71\x21\xd9\x43\x81\xc3\xba\x90\x13\xeb\xef\x2f\xb8\xb0\xbf\x9f\x08\x1f\x96\xec\xf1\x3c\xfa\xd0\x4e\x44\xc1\x1e\xbb\x35\x81\x60\xa8\x90\x49\xbf\xad\x5e\x8e\x24\x1d\x71\x68\x9d\xde\xec\xff\x02\x78\x06\x3f\xd8\x6b\x0a\xd4\x75\xc6\xa2\x52\x65\xf5\x56\xb3\x0d\xdb\x50\x07\x8e\x21\x62\x67\xed\xcd\x4a\x2b\x70\x16\x34\x5d\x4b\x76\x80\x6d\x7b\x02\xc6\x25\xf3\xf7\x17\xe0\xf6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb9\x4d\xeb\xad\xc8\x33\xd5\x70\x6c\xd4\x73\x6b\xb1\xdc\x75\x03\x98\x27\x83\x2a\xe4\x08\x85\x9e\x2e\x6a\x69\x41"[..]);

        ctx.update(b"\xeb\x71\xb4\x5a\x49\x4e\x76\x46\x2e\xdf\x41\xa9\xfd\xcb\xb3\xf4\x6f\xb8\x63\xb9\xe2\x59\xd0\xc8\xf4\xa7\x98\x98\x51\x6e\xeb\xe8\xc9\x0c\x3e\xa5\xa6\x75\x44\x0f\x3c\x7b\x1a\x18\xc1\x4d\xc2\x0c\x5f\x3d\xd2\x77\x88\xc6\x6d\x44\x8a\xcd\x73\x22\x63\x27\xf5\x2c\xd6\x5c\xec\xc8\xbe\xaa\x2a\xcf\xa3\x4d\x90\xef\x8b\xfe\x82\x4e\x12\xba\x98\x70\xbd\xc4\x96\x5b\x8c\xed\x9f\xf9\xce\x13\xa5\xbd\x39\xe8\x24\x89\x3a\xf4\x10\xd0\x8a\xde\x0c\xf8\x02\xe7\xdc\x02\xb0\xb7\x1d\x6c\x2a\x5c\x33\x56\x22\x90\x84\xe5\x3b\x3a\xe4\xe5\x1b\x38\x4f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfb\xbe\xc0\x5e\xe1\xfb\x5f\x5c\xd1\x10\x6e\xd7\x38\x48\x50\x05\x9c\xdc\xda\x47\x4b\xa7\xce\xc0\x40\x7a\x27\x2b"[..]);

        ctx.update(b"\x4e\xca\x0c\x51\xd3\x08\x29\xb9\xa1\xd2\x71\x2d\xa1\xfa\xc3\x1f\x52\x94\x2d\x77\xc9\xf2\x0c\x2b\xf6\xd3\x75\x10\x28\xd7\xd4\xf0\xd3\x36\xd3\xdc\x92\xb2\x7e\xc3\x68\xca\xa4\x44\x4b\x31\x80\xc1\xe3\x7e\x98\xb5\x8f\x25\xe6\x47\xa9\xa6\x36\x1f\x0b\x04\xcf\x78\xd1\x79\x55\x76\x61\x68\xee\xba\xa9\x93\xa4\x35\xa8\x8e\x0b\x39\x30\x74\x23\xd6\xea\xd8\x7f\x63\x9a\xfe\xa7\x5b\xa4\x4b\xbc\x6b\xd0\xfb\x5a\xc8\x4a\x12\xc2\xc6\xed\x95\x39\xa7\xc0\xf9\xab\xb0\xc1\xdc\x94\x83\xe2\xf3\x21\xa8\x52\x44\x92\x6d\xfd\x95\xe2\xf0\x56\x24\xaa\x7a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfe\x31\x3e\xb7\x4f\x95\x5c\x0c\xbb\x1c\x44\x6d\xd4\xff\x85\x3f\x32\xb3\x23\x2d\x93\xfa\xba\x7d\xb6\xd1\xfa\xb8"[..]);

        ctx.update(b"\x97\x78\x4d\x14\xdb\x62\xa7\xf9\x8f\x5a\xc3\xdf\x74\x2e\x01\x34\x89\xec\x0b\x87\x77\xb0\x5e\xf8\x2b\xba\x06\xed\xc5\xc3\xa8\x07\xb1\x91\xc6\x55\x13\xca\x3f\xc7\x69\x06\x15\xe5\x6c\x27\x73\xc0\x36\xed\xef\x29\xaa\xc5\x0c\x22\x11\xe2\x03\x92\x01\x8f\xc3\x3d\x83\xc4\x36\xf2\x74\xf7\xc6\x06\x2c\x34\x20\x02\x5e\x70\x37\x99\x3f\x1b\x8c\xdd\xeb\xf4\xae\xb2\x04\x21\xfc\x82\x9c\x7f\xb2\x32\x55\x37\x24\x55\xc6\x92\x44\xa0\x21\x0e\x6a\x9e\x13\xb1\x55\xa5\xec\x9d\x6d\x09\x00\xe5\x4a\x8f\x4d\x9f\x7a\x25\x5e\x3a\x7f\xd0\x6f\x12\x18\xe5\xd1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x55\x04\xf3\x91\x31\x77\x35\x50\xb6\xf4\x59\xf3\x3a\x5b\x57\xa2\xce\x60\xce\x8b\xb7\x8c\x57\x4f\xef\x83\xdc\xf7"[..]);

        ctx.update(b"\x1e\xe9\x04\x73\x51\xe2\xa1\x3e\x4a\x2d\x5a\x82\x6e\x30\x4f\xef\x82\x24\x1f\xba\xb5\x10\x08\x35\xe1\xf8\x50\xa2\x0e\x51\xe3\x49\x38\xb9\x3d\xc8\x52\xe5\x8a\xab\x8a\xdb\x0c\x3c\xcf\x61\xbe\x9c\x90\xb5\x37\x13\xc7\x7e\xd0\xa5\x37\x03\x09\xe6\xf1\x9b\x29\x0f\x1d\x64\x25\x50\xf7\x38\xc3\x68\x18\xdd\xff\x74\xf7\x7c\xae\x04\xaf\x55\x61\x74\x03\xb0\x8c\x7a\x9f\x17\xe8\xfb\xa0\xc2\x15\x23\x57\x53\x84\xb4\x4a\xc4\x94\x9e\x7c\x9d\xfb\xd1\xef\x6a\x68\x4f\x66\x6c\x67\x85\x6f\x8f\x84\xdb\xa1\x9c\xb3\x8a\x23\xb0\xef\xad\x6e\xed\x22\x9c\x53\x6f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb8\xf2\x53\x51\x2d\xab\xf9\xd8\x9d\x20\x80\x83\x0f\x23\xda\x58\x93\xb0\xf8\x7e\xdc\x0b\xd6\x24\xea\x76\x7f\x14"[..]);

        ctx.update(b"\x1f\x36\x3d\x2f\x7a\xa8\x9e\x2b\x6c\x5e\x17\x2f\x53\x0d\x1a\x35\x53\x1d\x00\x83\xa5\xac\xfc\xd2\x32\xd6\x4d\xb0\x61\x34\xb8\x23\x2d\xa2\x36\x8f\x7a\x46\xea\xd9\xa9\xce\x55\xcd\x6a\xf8\xcd\xbd\xd1\x58\x2b\x6b\xad\x56\xc5\x2a\x15\x76\x9c\x3f\x43\xdc\xd6\x8d\xa6\x0f\x6e\x72\x32\xfd\x2a\xec\xfb\x3f\xcd\x00\x02\x9f\x8e\x5c\x4e\xd7\xca\x3b\x3f\x9c\xf6\x89\x20\xdb\xd7\x47\xfb\x43\xf5\x32\xb1\x03\x4d\x9f\x49\xd5\x46\xaa\x89\x3b\xe6\x8f\xc3\x08\x46\x58\xf2\x23\x43\xb9\x06\x88\x77\x38\x7b\x8f\x68\x90\x30\x71\xfe\x58\x77\x08\x3b\xe0\x68\xd6\x26");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe5\x9a\x19\x68\x6d\xf3\x6b\xf5\xfe\x79\x8a\x95\x65\x72\x2b\x8e\x0b\xdd\x9f\x8e\xed\xbb\xb4\xa3\x4a\x9c\xa7\xab"[..]);

        ctx.update(b"\xec\xf5\xd9\xe2\x9c\x1c\x04\xc1\x1a\x95\x03\xcc\x22\x3d\x0c\xee\x48\x66\xfa\x26\xdf\x2b\x4f\x7c\x1a\x01\x79\x39\x71\x8f\x54\x57\x46\xc0\xf1\x37\xc9\x16\x96\x92\x19\x41\x05\xb2\xac\xf0\x01\xe2\xf0\xe7\x0f\x23\x32\x51\x7a\x20\xc0\x58\x99\x64\x4a\xf4\x54\xcb\x8e\x00\xe5\x36\x35\x93\xdc\x83\xf7\x8d\x66\xbd\x06\x70\xce\x8f\xaa\x72\x44\xff\x28\xd0\xde\x59\xe9\x64\xdc\x68\xd8\x7a\x30\xec\x0c\xe0\x3e\x49\xa7\x3c\xe0\x7d\xfe\xa2\xad\x54\xfa\x66\x7b\xdf\xbe\x2f\x22\x22\x89\x4d\x83\x0d\xde\x4d\xc9\xae\xe3\xca\xef\xa4\x08\x86\x83\xd7\xe8\xb9\xa9\x66");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa8\x86\xeb\x94\xf1\x5d\xf2\x08\xbe\x12\x29\x12\xd4\xed\xf0\x25\x61\x48\x22\x78\xa9\xf8\x47\xdd\xc9\x1c\x9b\xd2"[..]);

        ctx.update(b"\x9f\x44\x35\x76\x64\xb5\xe3\xa9\x58\x78\x06\x41\xcc\xa5\x20\x49\xf3\xb4\x9f\x07\x48\x4b\x5f\x76\x2a\x55\x71\xf7\xc9\x54\x1b\x43\x46\xf8\x1f\xa4\x16\xf0\x40\x65\xa8\x00\x03\x86\x47\x54\xb3\xb5\x41\x14\xa7\x7a\x49\x38\xc8\xb2\x1a\x9e\x4d\x3e\x5d\x59\xc9\xfc\xcd\x4d\x68\xf6\x99\xf9\x75\xda\x09\x93\x20\xab\x65\x5a\x7f\xb5\x13\x28\xd2\xc6\xff\x46\x0b\x9b\x40\x85\x8e\x99\xf8\x8a\x35\xbe\x7b\x6a\x97\xd6\xb4\x77\x8a\xf2\xc5\x59\xe6\x16\xee\x60\x8c\x32\xb0\x18\xa7\x53\x32\x1e\x32\x1b\xe3\x33\xbb\x6f\x61\x8f\x66\x6f\x9a\x77\x34\xab\x31\x12\x85\x93\x23");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x88\x39\xf7\x55\xee\xe8\x4e\x15\xc5\x86\xb5\x2e\x29\xa4\x1d\xdc\x64\x0a\xc4\x32\xcf\x31\x37\x06\x80\x98\x7a\x44"[..]);

        ctx.update(b"\xc1\xaa\x12\x66\xf2\x23\xc1\x48\xbf\xa3\xd0\xab\x29\xf2\x78\x33\x4d\x8f\xcb\xfb\xf0\xf4\xeb\xef\x5c\x1b\x7a\x76\x6b\x41\x51\x55\xe1\xea\x75\xd0\xfe\x25\x46\x11\x54\x11\xfa\xce\xd7\xa0\x4a\x27\x33\x9b\x6b\xcd\x62\xe7\x40\x69\x7d\x06\xce\x3c\xd2\xe0\xf0\x02\x38\xc4\x4c\x1d\x9f\xaa\x85\xef\xeb\xbb\xb3\x88\x03\x13\x10\x81\x24\xc5\xf3\x27\x7c\x1f\x03\xdd\xf4\x30\xa4\xbb\x4d\x88\xb6\x7b\x6e\x3f\x7f\x96\xfc\x39\xe5\xaa\x2c\xa7\xe1\x1f\xd5\xd1\x30\x0a\xca\x14\x4c\x51\x66\x26\x9a\x11\x68\xa2\xe5\x3c\x01\xc0\x0b\x87\x2c\x63\xf6\x83\x3e\x5a\xce\x09\xbe\xdf");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x43\x9e\x3c\x7a\x0d\x65\x5a\x30\xa9\x74\x9a\xfd\xef\xb7\xe0\x48\x81\x43\x35\x84\x9d\xf7\x6d\x52\x6c\x28\x77\x27"[..]);

        ctx.update(b"\x0a\x36\x7d\x37\x89\x82\x7c\xcd\x4b\xef\x5f\xe8\xeb\x78\xc2\x05\x03\x24\x1f\x07\xfb\x8c\x41\xd8\x1e\x97\xfb\x53\xf3\x89\x19\x62\xca\x3c\x97\x63\x95\xac\x11\xd1\xf9\xba\x7b\x20\xa5\x29\x12\xe8\xe3\xed\x92\x46\x6c\xa5\xaa\x80\x81\x66\xad\xe7\x37\xba\x8a\x02\x13\xe8\xfe\xe8\xd6\x76\x08\xee\x9a\xed\x9e\x82\x1e\xdc\x9e\x57\x5f\x1f\x07\xc3\x68\x61\x69\x65\x6a\xe0\x9a\x0a\x0f\x70\xab\xd1\x0c\xc3\x1a\x8e\xf6\xe7\x49\x6d\x56\x10\x2f\xd8\xff\x98\x4e\x9a\x9f\x44\xe5\x44\x95\xc9\x66\xcf\x02\x8f\x2a\x84\x23\xb4\x64\x19\xde\x54\x54\x1d\x9a\x08\xbd\x96\x54\xac\x98");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x40\x31\x80\x36\xa5\x95\x63\x0e\x41\x35\xf1\x07\x03\xbe\x1d\x75\x9a\x6c\x7e\x51\x46\xe0\xfc\x82\xab\xeb\xa1\x84"[..]);

        ctx.update(b"\x8a\x05\xb0\x0a\xe2\xd5\xf6\x52\xf0\x2f\x98\xa1\xb0\x35\x00\x3f\x8f\xa7\xba\x1b\x17\xfc\x37\x78\xcd\xb1\xca\xe3\x5a\xe1\xf7\x68\xea\x16\xed\x05\xd2\x5f\x51\x5f\x75\xa2\x3d\xb4\x68\x34\x89\x11\xd4\xa7\x49\xc5\x1c\xe3\x96\x15\xc0\x78\x92\x31\x82\x33\xa6\x67\xc7\xf0\x0e\x97\x3f\xae\x98\xe7\xc8\xe9\xa8\xb7\x90\x24\x80\xd8\x7a\xc5\xbe\xf8\xc4\x25\x26\x61\xe6\xe8\xa2\xe4\xbd\x8a\x87\x0f\xe8\x3b\x1a\xa7\x73\xed\x53\x52\xb2\xab\xe1\x93\x70\x2c\x6d\xfb\x4a\xa8\x23\x9e\x55\xea\x6f\xc5\x07\xa7\x04\xe2\x54\x0e\x23\xc9\x17\xa0\x1a\x1c\xb4\x42\x0b\x07\xfb\x90\xee\x2e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9a\x26\xf0\x54\xe5\x7a\xea\x14\x24\x2d\x78\x01\xf3\xd6\x1d\xdc\xa1\x52\x3b\x73\x8f\xc2\x6f\xec\xfa\x5d\x9a\x6a"[..]);

        ctx.update(b"\xba\x64\x42\xc6\xd2\x13\x92\x01\xdf\xef\x32\xc1\xff\xb0\xce\x92\xdd\x64\x09\x1b\xd5\x07\xc2\x50\x59\x53\x95\xe9\x93\xd9\xa5\x12\x4b\x51\x99\x64\x0c\x2f\xe5\x14\x82\x77\x4b\x6a\x27\xd1\xa1\x75\x1f\xe0\xd4\xfe\x5f\xd0\x2d\xba\x15\x2e\xd3\xc3\x44\xfd\x92\x49\xaf\x06\xda\x85\xf9\x6f\x0b\xef\x0a\x8f\xef\xb1\xb5\x01\x88\x5b\x97\xf7\x0d\xd8\x42\xd1\x2f\xa1\x9b\xef\xa0\x30\x80\xc3\xd6\xb8\xae\x2a\x0d\x13\xe2\xfc\x8b\xfc\x3f\xe1\x27\x7e\xf0\x67\x0c\xac\x0e\x52\xbb\x93\xc4\x34\x4f\x6d\xb1\x3d\x05\x18\x8d\x53\xfb\xc6\x10\x65\x38\xf5\x0f\xfd\xed\xa2\xe9\x15\xfa\xb9\x21");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x58\x47\x0d\xa5\x84\x76\xbc\xb8\x94\x50\xc5\x21\xfc\x39\x6c\x6d\xc5\x1b\x9f\xb6\x46\x5c\x97\x9a\xba\x5f\x8e\xb4"[..]);

        ctx.update(b"\x96\xfd\xb7\x6f\x83\xbf\x12\xb3\xf4\xf3\x22\xbf\x61\x3f\xc3\x8b\x2c\x8e\x06\x78\x85\x62\x30\x41\x8b\x6b\x06\x2f\xb3\x58\x48\x8d\x6e\xed\x7c\x5c\x06\x56\xec\x48\xc9\xbb\xf2\xda\x6a\x14\x73\xee\xa4\x3f\xaa\x68\x20\x4f\x27\x23\x99\x28\x17\x2a\x3e\x49\xc5\x2b\x58\xe8\x61\x28\x2c\x44\x01\x70\x23\x37\xe5\xce\x28\x0a\xff\x00\x52\x8e\xb2\x6a\xc3\x68\xdb\x0c\xd0\xad\x0e\xb2\x62\xaf\x22\x6a\x9b\x16\xef\x3b\xbd\x32\x56\x14\x48\x8f\x82\x03\x63\xca\x6e\xa7\x7d\xa4\xa7\xe8\x34\x55\x54\xe5\x76\x23\x73\x2e\xe6\x32\x65\x34\x81\x9e\xad\xfe\x81\xc7\xf5\x1d\x81\xec\x51\xe1\xe3\xfc");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xbe\x92\xd4\xa6\x94\x6d\xe0\xe9\x3d\x5b\xbe\x42\x06\x51\xa8\xbe\xfb\x97\xcb\xdb\x5d\x63\xb2\x2a\xae\xcf\x45\x3d"[..]);

        ctx.update(b"\x0e\xef\x94\x7f\x1e\x4f\x01\xcd\xb5\x48\x1c\xa6\xea\xa2\x5f\x2c\xac\xa4\xc4\x01\x61\x28\x88\xfe\xce\xf5\x2e\x28\x37\x48\xc8\xdf\xc7\xb4\x72\x59\x32\x2c\x1f\x4f\x98\x5f\x98\xf6\xad\x44\xc1\x31\x17\xf5\x1e\x05\x17\xc0\x97\x4d\x6c\x7b\x78\xaf\x74\x19\xbc\xce\x95\x7b\x8b\xc1\xdb\x88\x01\xc5\xe2\x80\x31\x2e\xf7\x8d\x6a\xa4\x7a\x9c\xb9\x8b\x86\x6a\xae\xc3\xd5\xe2\x63\x92\xdd\xa6\xbb\xde\x3f\xec\xe8\xa0\x62\x8b\x30\x95\x5b\x55\xf0\x37\x11\xa8\xe1\xeb\x9e\x40\x9a\x7c\xf8\x4f\x56\xc8\xd0\xd0\xf8\xb9\xba\x18\x4c\x77\x8f\xae\x90\xdc\x0f\x5c\x33\x29\xcb\x86\xdc\xf7\x43\xbb\xae");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x98\xec\x52\xc2\x1c\xb9\x88\xb1\x43\x4b\x16\x53\xdd\x4a\xc8\x06\xd1\x18\xde\x6a\xf1\xbb\x47\x1c\x16\x57\x7c\x34"[..]);

        ctx.update(b"\xe6\x5d\xe9\x1f\xdc\xb7\x60\x6f\x14\xdb\xcf\xc9\x4c\x9c\x94\xa5\x72\x40\xa6\xb2\xc3\x1e\xd4\x10\x34\x6c\x4d\xc0\x11\x52\x65\x59\xe4\x42\x96\xfc\x98\x8c\xc5\x89\xde\x2d\xc7\x13\xd0\xe8\x24\x92\xd4\x99\x1b\xd8\xc4\xc5\xe6\xc7\x4c\x75\x3f\xc0\x93\x45\x22\x5e\x1d\xb8\xd5\x65\xf0\xce\x26\xf5\xf5\xd9\xf4\x04\xa2\x8c\xf0\x0b\xd6\x55\xa5\xfe\x04\xed\xb6\x82\x94\x2d\x67\x5b\x86\x23\x5f\x23\x59\x65\xad\x42\x2b\xa5\x08\x1a\x21\x86\x5b\x82\x09\xae\x81\x76\x3e\x1c\x4c\x0c\xcc\xbc\xcd\xaa\xd5\x39\xcf\x77\x34\x13\xa5\x0f\x5f\xf1\x26\x7b\x92\x38\xf5\x60\x2a\xdc\x06\x76\x4f\x77\x5d\x3c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x26\xec\x9d\xf5\x4d\x9a\xfe\x11\x71\x07\x72\xbf\xbe\xcc\xc8\x3d\x9d\x04\x39\xd3\x53\x07\x77\xc8\x1b\x8a\xe6\xa3"[..]);
    }
}
