//go:build integration

/*
Copyright 2020 The Kubernetes Authors All rights reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package integration

import (
	"context"
	"fmt"
	"os/exec"
	"strings"
	"testing"

	"k8s.io/minikube/pkg/drivers/kic/oci"
)

// TestKicCustomNetwork verifies the docker driver works with a custom network
func TestKicCustomNetwork(t *testing.T) {
	if !KicDriver() {
		t.Skip("only runs with docker driver")
	}

	tests := []struct {
		description string
		networkName string
	}{
		{
			description: "create custom network",
		}, {
			description: "use default bridge network",
			networkName: "bridge",
		},
	}

	for _, test := range tests {
		t.Run(test.description, func(t *testing.T) {
			profile := UniqueProfileName("docker-network")
			ctx, cancel := context.WithTimeout(context.Background(), Minutes(5))
			defer Cleanup(t, profile, cancel)

			startArgs := []string{"start", "-p", profile, fmt.Sprintf("--network=%s", test.networkName)}
			c := exec.CommandContext(ctx, Target(), startArgs...)
			rr, err := Run(t, c)
			if err != nil {
				t.Fatalf("%v failed: %v\n%v", rr.Command(), err, rr.Output())
			}
			nn := test.networkName
			if nn == "" {
				nn = profile
			}
			verifyNetworkExists(ctx, t, nn)
		})
	}
}

// TestKicExistingNetwork verifies the docker driver and run with an existing network
func TestKicExistingNetwork(t *testing.T) {
	if !KicDriver() {
		t.Skip("only runs with docker driver")
	}
	// create custom network
	networkName := "existing-network"
	if _, err := oci.CreateNetwork(oci.Docker, networkName, "", ""); err != nil {
		t.Fatalf("error creating network: %v", err)
	}
	defer func() {
		if err := oci.DeleteKICNetworksByLabel(oci.Docker, networkName); err != nil {
			t.Logf("error deleting kic network, may need to delete manually: %v", err)
		}
	}()
	profile := UniqueProfileName("existing-network")
	ctx, cancel := context.WithTimeout(context.Background(), Minutes(5))
	defer Cleanup(t, profile, cancel)

	verifyNetworkExists(ctx, t, networkName)

	startArgs := []string{"start", "-p", profile, fmt.Sprintf("--network=%s", networkName)}
	c := exec.CommandContext(ctx, Target(), startArgs...)
	rr, err := Run(t, c)
	if err != nil {
		t.Fatalf("%v failed: %v\n%v", rr.Command(), err, rr.Output())
	}
}

// TestKicCustomSubnet verifies the docker/podman driver works with a custom subnet
func TestKicCustomSubnet(t *testing.T) {
	if !KicDriver() {
		t.Skip("only runs with docker/podman driver")
	}

	profile := UniqueProfileName("custom-subnet")
	ctx, cancel := context.WithTimeout(context.Background(), Minutes(5))
	defer Cleanup(t, profile, cancel)

	subnet := "192.168.60.0/24"
	startArgs := []string{"start", "-p", profile, fmt.Sprintf("--subnet=%s", subnet)}
	c := exec.CommandContext(ctx, Target(), startArgs...)
	rr, err := Run(t, c)
	if err != nil {
		t.Fatalf("%v failed: %v\n%v", rr.Command(), err, rr.Output())
	}

	verifySubnet(ctx, t, profile, subnet)
}

// TestKicStaticIP starts minikube with the static IP flag
func TestKicStaticIP(t *testing.T) {
	if !KicDriver() {
		t.Skip("only run with docker/podman driver")
	}
	profile := UniqueProfileName("static-ip")
	ctx, cancel := context.WithTimeout(context.Background(), Minutes(5))
	defer Cleanup(t, profile, cancel)

	staticIP := "192.168.200.200"
	startArgs := []string{"start", "-p", profile, fmt.Sprintf("--static-ip=%s", staticIP)}
	c := exec.CommandContext(ctx, Target(), startArgs...)
	rr, err := Run(t, c)
	if err != nil {
		t.Fatalf("%v failed: %v\n%v", rr.Command(), err, rr.Output())
	}

	c = exec.CommandContext(ctx, Target(), "-p", profile, "ip")
	rr, err = Run(t, c)
	if err != nil {
		t.Fatalf("%s failed: %v\n%s", rr.Command(), err, rr.Output())
	}

	if !strings.Contains(rr.Output(), staticIP) {
		t.Errorf("static IP (%s) not found in output %s", staticIP, rr.Output())
	}
}

func verifyNetworkExists(ctx context.Context, t *testing.T, networkName string) {
	c := exec.CommandContext(ctx, "docker", "network", "ls", "--format", "{{.Name}}")
	rr, err := Run(t, c)
	if err != nil {
		t.Fatalf("%v failed: %v\n%v", rr.Command(), err, rr.Output())
	}
	if output := rr.Output(); !strings.Contains(output, networkName) {
		t.Fatalf("%s network is not listed by [%v]: %v", networkName, c.Args, output)
	}
}

func verifySubnet(ctx context.Context, t *testing.T, network, subnet string) {
	c := exec.CommandContext(ctx, "docker", "network", "inspect", network, "--format", "{{(index .IPAM.Config 0).Subnet}}")
	rr, err := Run(t, c)
	if err != nil {
		t.Fatalf("%v failed: %v\n%v", rr.Command(), err, rr.Output())
	}

	if output := strings.TrimSpace(rr.Output()); !strings.Contains(output, subnet) {
		t.Fatalf("%s subnet not match to %v", subnet, output)
	}
}
