#include <vector>
#include <iostream>
#include <fstream>
#include <thread>
#include <fcntl.h>
#include <cstdio>
#include <random>
#include <sys/stat.h>

#include "platform.hh"

#if (BOOST_OS_WINDOWS)
#include <io.h>
#include <Windows.h>
#else
#include <unistd.h>
#endif //#if BOOST_OS_WINDOWS

#include "common.hh"
#include "game/image.hh"

const std::vector<uint8_t> PNG_IMAGE{ 0x89,0x50,0x4e,0x47,0x0d,0x0a,0x1a,0x0a,0x00,0x00,0x00,0x0d,0x49,0x48,0x44,0x52,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x01,0x08,0x02,0x00,0x00,0x00,0x90,0x77,0x53,0xde,0x00,0x00,0x00,0x0c,0x49,0x44,0x41,0x54,0x08,0xd7,0x63,0x58,0xca,0xc0,0x00,0x00,0x01,0xf3,0x00,0xa6,0x8e,0x7e,0x59,0x91,0x00,0x00,0x00,0x00,0x49,0x45,0x4e,0x44,0xae,0x42,0x60,0x82 };

const std::vector<uint8_t> JPEG_IMAGE{ 0xff,0xd8,0xff,0xe0,0x00,0x10,0x4a,0x46,0x49,0x46,0x00,0x01,0x01,0x01,0x01,0x2c,0x01,0x2c,0x00,0x00,0xff,0xdb,0x00,0x43,0x00,0x03,0x02,0x02,0x03,0x02,0x02,0x03,0x03,0x03,0x03,0x04,0x03,0x03,0x04,0x05,0x08,0x05,0x05,0x04,0x04,0x05,0x0a,0x07,0x07,0x06,0x08,0x0c,0x0a,0x0c,0x0c,0x0b,0x0a,0x0b,0x0b,0x0d,0x0e,0x12,0x10,0x0d,0x0e,0x11,0x0e,0x0b,0x0b,0x10,0x16,0x10,0x11,0x13,0x14,0x15,0x15,0x15,0x0c,0x0f,0x17,0x18,0x16,0x14,0x18,0x12,0x14,0x15,0x14,0xff,0xdb,0x00,0x43,0x01,0x03,0x04,0x04,0x05,0x04,0x05,0x09,0x05,0x05,0x09,0x14,0x0d,0x0b,0x0d,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0x14,0xff,0xc0,0x00,0x11,0x08,0x00,0x01,0x00,0x01,0x03,0x01,0x11,0x00,0x02,0x11,0x01,0x03,0x11,0x01,0xff,0xc4,0x00,0x14,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0xff,0xc4,0x00,0x14,0x10,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0xc4,0x00,0x15,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x07,0x08,0xff,0xc4,0x00,0x14,0x11,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0xda,0x00,0x0c,0x03,0x01,0x00,0x02,0x11,0x03,0x11,0x00,0x3f,0x00,0x16,0x0d,0x16,0xdb,0xff,0xd9 };

const std::vector<uint8_t> WEBP_IMAGE{ 0x52,0x49,0x46,0x46,0x40,0x00,0x00,0x00,0x57,0x45,0x42,0x50,0x56,0x50,0x38,0x20,0x34,0x00,0x00,0x00,0xd0,0x01,0x00,0x9d,0x01,0x2a,0x01,0x00,0x01,0x00,0x00,0xc0,0x12,0x25,0xa0,0x02,0x74,0xba,0x01,0xf8,0x00,0x03,0xb0,0x00,0xfe,0xfd,0x69,0x7b,0xff,0xd3,0x48,0xf1,0xa4,0x78,0xd2,0x3e,0x62,0xdf,0xfe,0xdf,0x0f,0xf7,0x1e,0x5f,0xb7,0xc3,0xff,0xdb,0xb2,0x00,0x00,0x00 };


const std::vector<uint8_t> SVG_IMAGE{ 0x3c,0x3f,0x78,0x6d,0x6c,0x20,0x76,0x65,0x72,0x73,0x69,0x6f,0x6e,0x3d,0x22,0x31,0x2e,0x30,0x22,0x20,0x65,0x6e,0x63,0x6f,0x64,0x69,0x6e,0x67,0x3d,0x22,0x55,0x54,0x46,0x2d,0x38,0x22,0x20,0x73,0x74,0x61,0x6e,0x64,0x61,0x6c,0x6f,0x6e,0x65,0x3d,0x22,0x79,0x65,0x73,0x22,0x3f,0x3e,0x0d,0x0a,0x3c,0x21,0x44,0x4f,0x43,0x54,0x59,0x50,0x45,0x20,0x73,0x76,0x67,0x20,0x50,0x55,0x42,0x4c,0x49,0x43,0x20,0x22,0x2d,0x2f,0x2f,0x57,0x33,0x43,0x2f,0x2f,0x44,0x54,0x44,0x20,0x53,0x56,0x47,0x20,0x31,0x2e,0x31,0x2f,0x2f,0x45,0x4e,0x22,0x20,0x22,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x77,0x77,0x77,0x2e,0x77,0x33,0x2e,0x6f,0x72,0x67,0x2f,0x47,0x72,0x61,0x70,0x68,0x69,0x63,0x73,0x2f,0x53,0x56,0x47,0x2f,0x31,0x2e,0x31,0x2f,0x44,0x54,0x44,0x2f,0x73,0x76,0x67,0x31,0x31,0x2e,0x64,0x74,0x64,0x22,0x3e,0x0d,0x0a,0x3c,0x73,0x76,0x67,0x20,0x77,0x69,0x64,0x74,0x68,0x3d,0x22,0x34,0x30,0x30,0x22,0x20,0x68,0x65,0x69,0x67,0x68,0x74,0x3d,0x22,0x34,0x30,0x30,0x22,0x20,0x76,0x69,0x65,0x77,0x42,0x6f,0x78,0x3d,0x22,0x30,0x20,0x30,0x20,0x32,0x30,0x30,0x20,0x32,0x30,0x30,0x22,0x20,0x78,0x6d,0x6c,0x6e,0x73,0x3d,0x22,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x77,0x77,0x77,0x2e,0x77,0x33,0x2e,0x6f,0x72,0x67,0x2f,0x32,0x30,0x30,0x30,0x2f,0x73,0x76,0x67,0x22,0x3e,0x0d,0x0a,0x20,0x20,0x3c,0x70,0x61,0x74,0x68,0x20,0x66,0x69,0x6c,0x6c,0x3d,0x22,0x72,0x65,0x64,0x22,0x20,0x73,0x74,0x72,0x6f,0x6b,0x65,0x3d,0x22,0x23,0x30,0x30,0x30,0x22,0x20,0x73,0x74,0x72,0x6f,0x6b,0x65,0x2d,0x77,0x69,0x64,0x74,0x68,0x3d,0x22,0x32,0x2e,0x35,0x22,0x20,0x64,0x3d,0x22,0x4d,0x31,0x36,0x38,0x20,0x31,0x36,0x38,0x48,0x33,0x32,0x56,0x33,0x32,0x68,0x31,0x33,0x36,0x7a,0x22,0x2f,0x3e,0x0d,0x0a,0x3c,0x2f,0x73,0x76,0x67,0x3e,0x0d,0x0a };

const std::vector<uint8_t> PDF_IMAGE{ 0x25,0x50,0x44,0x46,0x2d,0x31,0x2e,0x35,0x0a,0x25,0xb5,0xed,0xae,0xfb,0x0a,0x34,0x20,0x30,0x20,0x6f,0x62,0x6a,0x0a,0x3c,0x3c,0x20,0x2f,0x4c,0x65,0x6e,0x67,0x74,0x68,0x20,0x35,0x20,0x30,0x20,0x52,0x0a,0x20,0x20,0x20,0x2f,0x46,0x69,0x6c,0x74,0x65,0x72,0x20,0x2f,0x46,0x6c,0x61,0x74,0x65,0x44,0x65,0x63,0x6f,0x64,0x65,0x0a,0x3e,0x3e,0x0a,0x73,0x74,0x72,0x65,0x61,0x6d,0x0a,0x78,0x9c,0x33,0x54,0x30,0x00,0x42,0x5d,0x43,0x20,0x61,0xa8,0x90,0x9c,0xcb,0x55,0xc8,0x65,0xa0,0x67,0x66,0x62,0x6e }; // just the first few bytes, tail truncated

const std::vector<uint8_t> SMALL_FILE{ 0x01, 0x02, 0x03 };
const std::vector<uint8_t> EMPTY_FILE{ };

std::string getSecureTmpFile()
{
#if (BOOST_OS_WINDOWS)
    char tempPath[MAX_PATH];
    char tempFile[MAX_PATH];
    DWORD pathLen = GetTempPathA(MAX_PATH, tempPath);
    if (pathLen > 0 && pathLen < MAX_PATH)
    {
        if (GetTempFileNameA(tempPath, "performous", 0, tempFile))
        {
            return std::string(tempFile);
        }
    }
#else
    std::string charset = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
    std::random_device rd;
    std::mt19937 gen(rd());
    std::uniform_int_distribution<> dist(0, charset.size() - 1);

    for (int attempt = 0; attempt < 100; ++attempt) {
        std::string name = ( std::filesystem::temp_directory_path() / "performous-unittest-" );
        for (int i = 0; i < 8; ++i)
            name += charset[dist(gen)];

        int fd = open(name.c_str(), O_CREAT | O_EXCL | O_RDWR, 0600);
        if (fd != -1) {
            close(fd);
            return name;
        }
    }
#endif // #ifdef BOOST_OS_WINDOWS

    return std::string();  // fail
}

// create a temporary test file
std::string tmpTestFile( const std::vector<uint8_t> &image) {
    std::string tmp_name = getSecureTmpFile();
    try {
        std::ofstream fout(tmp_name, std::ios::binary);
        fout.write((char*)image.data(), image.size());
        fout.close();
        return tmp_name;
    }
    catch (...) {  // yes, catch _everything_, on purpose.
        return "";
    }
}

TEST(ImageType, PNGImageDetect) {
    std::string filename = tmpTestFile(PNG_IMAGE);

    EXPECT_TRUE(filename != "");
    EXPECT_TRUE(getImageType(filename) == ImageType::PNG);
}

TEST(ImageType, JPEGImageDetect) {
    std::string filename = tmpTestFile(JPEG_IMAGE);

    EXPECT_TRUE(filename != "");
    EXPECT_TRUE(getImageType(filename) == ImageType::JPEG);
}

TEST(ImageType, SVGImageDetect) {
    std::string filename = tmpTestFile(SVG_IMAGE);

    EXPECT_TRUE(filename != "");
    EXPECT_TRUE(getImageType(filename) == ImageType::SVG);
}

TEST(ImageType, WEBPImageDetect) {
    std::string filename = tmpTestFile(WEBP_IMAGE);

    EXPECT_TRUE(filename != "");
    EXPECT_TRUE(getImageType(filename) == ImageType::WEBP);
}

TEST(ImageType, PDFImageDetect) {
    std::string filename = tmpTestFile(PDF_IMAGE);

    EXPECT_TRUE(filename != "");
    EXPECT_TRUE(getImageType(filename) == ImageType::UNKNOWN);
}

TEST(ImageType, ShortImageDetect) {
    std::string filename = tmpTestFile(SMALL_FILE);

    EXPECT_TRUE(filename != "");
    EXPECT_TRUE(getImageType(filename) == ImageType::UNKNOWN);
}

TEST(ImageType, ZeroImageDetect) {
    std::string filename = tmpTestFile(EMPTY_FILE);

    EXPECT_TRUE(filename != "");
    EXPECT_TRUE(getImageType(filename) == ImageType::UNKNOWN);
}

